import itertools
import os
import os.path
import random

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import task

from sandbox.projects import resource_types
from sandbox.projects.common import dolbilka
from sandbox.projects.common import string
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.images.resources import fuzzing


_SHARD_INSTANCE_ATTR = search_settings.SHARD_INSTANCE_ATTRIBUTE_NAME


class DatabaseParameter(parameters.LastReleasedResource):
    name = 'database_resource_id'
    description = 'Database:'
    resource_type = [resource_types.IMAGES_SEARCH_DATABASE]


class RequestLimitParameter(parameters.SandboxIntegerParameter):
    name = 'request_limit'
    description = 'Unique requests number upper limit:'
    default_value = 100000


class FuzzingRequestLimitParameter(parameters.SandboxIntegerParameter):
    name = 'fuzzing_request_limit'
    description = 'Unique requests number upper limit for fuzzing tests:'
    default_value = 100


class AttributesParameter(parameters.SandboxStringParameter):
    name = 'attributes'
    description = 'Set additional attrs to resources (ex.: attr1=v1, attr2=v2)'
    do_not_copy = True


class ImagesGenerateRimdaemonRequests(fuzzing.GenerateFuzzingTask, task.SandboxTask):
    """
        Dumps rim databse to build http-requests to rimdaemon
    """
    type = 'IMAGES_GENERATE_RIMDAEMON_REQUESTS'
    execution_space = 60 * 1024

    input_parameters = (
        DatabaseParameter,
        RequestLimitParameter,
        AttributesParameter,
        FuzzingRequestLimitParameter,
    ) + fuzzing.GenerateFuzzingTask.input_parameters

    def __get_ids(self, db_resource_id):
        db_path = self.sync_resource(db_resource_id)
        alive_path = os.path.join(db_path, "rimdb.alive")
        if not os.path.exists(alive_path):
            raise errors.SandboxTaskFailureError("No {} in RIM database".format(alive_path))
        with open(alive_path, "r") as f:
            for line in f:
                yield line.strip()

    def __get_requests_path(self):
        return self.abs_path('rim-requests.txt')

    def __get_plan_path(self):
        return self.abs_path('rim-requests.plan')

    def _fuzzing_dir(self):
        return "extsearch/images/daemons/rimdaemon/lib/fuzzing"

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        fuzzing.GenerateFuzzingTask.on_enqueue(self)

        database_resource = channel.sandbox.get_resource(self.ctx[DatabaseParameter.name])
        attributes = string.parse_attrs(self.ctx[AttributesParameter.name])
        if _SHARD_INSTANCE_ATTR in database_resource.attributes:
            attributes.update({_SHARD_INSTANCE_ATTR: database_resource.attributes[_SHARD_INSTANCE_ATTR]})
        self.create_resource(
            self.descr,
            self.__get_requests_path(),
            resource_types.PLAIN_TEXT_QUERIES,
            arch=sandboxapi.ARCH_ANY,
            attributes=attributes
        )
        self.create_resource(
            self.descr,
            self.__get_plan_path(),
            resource_types.BASESEARCH_PLAN,
            arch=sandboxapi.ARCH_ANY,
            attributes=attributes
        )

    def on_execute(self):
        self.sync_resource(self.ctx[DatabaseParameter.name])
        database_resource = channel.sandbox.get_resource(self.ctx[DatabaseParameter.name])
        random.seed(database_resource.file_md5)

        with open(self.__get_requests_path(), 'w') as result_file:
            for doc_id in itertools.islice(self.__get_ids(database_resource),
                                           self.ctx[RequestLimitParameter.name]):
                query = "/rim?docid={}".format(doc_id)
                result_file.write(query + "\n")

        dolbilka.convert_queries_to_plan(self.__get_requests_path(), self.__get_plan_path())

        self._fuzzing_corpus_init()

        for doc_id in itertools.islice(self.__get_ids(database_resource),
                                       self.ctx[FuzzingRequestLimitParameter.name]):
            query = "/rim?docid={}".format(doc_id)
            self._fuzzing_corpus_add(query, int(self.ctx[FuzzingRequestLimitParameter.name]))

        self._fuzzing_corpus_finish(self.ctx[DatabaseParameter.name])


__Task__ = ImagesGenerateRimdaemonRequests
