import json
import logging
import tarfile
from os import listdir
from os import path

from sandbox import sdk2
import sandbox.sdk2.helpers as sdk2_helpers
import sandbox.common.types.resource as ctr
import sandbox.sandboxsdk.errors as errors
from sandbox.sandboxsdk import sandboxapi

from sandbox.projects import resource_types
from sandbox.projects.common import dolbilka
from sandbox.projects.common import string
from sandbox.projects.common.dolbilka import resources as dolbilka_resources
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.images.daemons import resources as daemons_resources


_SHARD_INSTANCE_ATTR = search_settings.SHARD_INSTANCE_ATTRIBUTE_NAME
_APPHOST_REQUEST_TYPE = "apphost-synthetic"


class ImagesGenerateRimpatchdaemonRequests(sdk2.Task):
    """
        Dumps rim http rim responses to build apphost-requests to rimpatchdaemon
    """

    class Parameters(sdk2.Task.Parameters):
        rim_response = sdk2.parameters.LastReleasedResource(
            'RIM responses',
            resource_type=[
                daemons_resources.IMAGES_RIM_RESPONSES,
                resource_types.THUMB_DAEMON_RESPONSES_ARCHIVE],
            state=(ctr.State.READY),
            required=True
        )
        request_limit = sdk2.parameters.Integer(
            'Requests number upper limit',
            default_value=100000
        )
        attributes = sdk2.parameters.String(
            'Set additional attrs to resources (ex.: attr1=v1, attr2=v2)',
            do_not_copy=True
        )

    class Requirements(sdk2.Task.Requirements):
        disk_space = 5 * 1024  # 5 Gb

    def on_execute(self):
        attributes = string.parse_attrs(self.Parameters.attributes)
        request_attributes = {k.format(_APPHOST_REQUEST_TYPE): v for k, v in attributes.iteritems()}

        if hasattr(self.Parameters.rim_response, _SHARD_INSTANCE_ATTR):
            request_attributes.update({
                _SHARD_INSTANCE_ATTR: getattr(self.Parameters.rim_response, _SHARD_INSTANCE_ATTR)
            })

        plain_text_queries_res = resource_types.PLAIN_TEXT_QUERIES(
            self,
            self.Parameters.description,
            'rimpatch-requests.txt'
        )
        dolbilka_queries_res = dolbilka_resources.DOLBILKA_STPD_QUERIES(
            self,
            self.Parameters.description,
            'stpd_queries.bin'
        )
        basesearch_plan_res = resource_types.BASESEARCH_PLAN(
            self,
            self.Parameters.description,
            'rimpatch-requests.plan'
        )
        for k in request_attributes:
            setattr(plain_text_queries_res, k, request_attributes[k])
            setattr(dolbilka_queries_res, k, request_attributes[k])
            setattr(basesearch_plan_res, k, request_attributes[k])

        plain_text_queries = str(sdk2.ResourceData(plain_text_queries_res).path)
        dolbilka_queries = str(sdk2.ResourceData(dolbilka_queries_res).path)
        basesearch_plan = str(sdk2.ResourceData(basesearch_plan_res).path)

        request_limit = self.Parameters.request_limit
        with open(str(plain_text_queries), 'w') as result_file:
            for rim_response in self._responses_iterator(self.Parameters.rim_response, request_limit):
                rim_response['type'] = 'images_rim_response'
                rimpatch_request = {
                    'answers': [{
                        'name': 'RIMDAEMON',
                        'results': [rim_response]
                    }]
                }
                result_file.write(json.dumps(rimpatch_request) + "\n")

        app_host_plan_builder = str(sdk2.ResourceData(
            resource_types.APP_HOST_TOOLS_BUNDLE.find(
                arc=sandboxapi.ARCH_LINUX, released='stable').order(-sdk2.Resource.id).first()
        ).path) + "/make_tank_ammo"

        with sdk2_helpers.ProcessLog(self, logger=logging.getLogger('apphost_plan_builder')) as pl:
            proc = sdk2_helpers.subprocess.Popen(
                [app_host_plan_builder, '-i', plain_text_queries, '-o', dolbilka_queries],
                stdout=sdk2_helpers.subprocess.PIPE,
                stderr=pl.stderr
            )
            stdout, _ = proc.communicate()
            rc = proc.wait()
            if rc:
                raise errors.SandboxTaskFailureError('Failed to get plan')

        # generate plans for dolbilka
        dolbilka.convert_queries_to_plan(
            dolbilka_queries,
            basesearch_plan,
            loader_type='phantom'
        )

    def _responses_iterator(self, rim_responses_resource, request_limit):
        request_no = 0
        if isinstance(rim_responses_resource, daemons_resources.IMAGES_RIM_RESPONSES):
            with open(str(sdk2.ResourceData(rim_responses_resource).path)) as responses_file:
                for line in responses_file:
                    fields = line.rstrip().split('\t')
                    if len(fields) < 3 or fields[1] != "200" or not fields[2]:
                        continue
                    yield json.loads(fields[2])
                    request_no += 1
                    if request_limit and request_no == request_limit:
                        break
        elif isinstance(rim_responses_resource, resource_types.THUMB_DAEMON_RESPONSES_ARCHIVE):
            with tarfile.open(str(sdk2.ResourceData(rim_responses_resource).path) + '/data.tar.gz') as tar:
                tar.extractall()
            for resp_file in listdir('data'):
                resp_file = 'data/' + resp_file
                if path.isfile(resp_file):
                    with open(resp_file) as fl:
                        try:
                            data = json.load(fl)
                            yield data
                        except ValueError:
                            # It's not rim-serp data
                            continue
                        request_no += 1
                        if request_limit and request_no == request_limit:
                            break
        else:
            raise errors.SandboxTaskFailureError(
                "Invalid resource '{}'. Expected IMAGES_RIM_RESPONSES or "
                "THUMB_DAEMON_RESPONSES_ARCHIVE".format(str(rim_responses_resource))
            )
