import logging
import os.path
import urllib2
import cStringIO as StringIO

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import sandboxapi

from sandbox.projects import resource_types
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
import sandbox.projects.common.app_host.converter as apphost_converter
from sandbox.projects.common.search import requester as search_requester
from sandbox.projects.common.search import requester_compat as search_requester_compat
from sandbox.projects.common.dolbilka import resources as dolbilka_resources
from sandbox.projects.images.daemons import task as daemons_task
from sandbox.projects.images.daemons import resources as daemons_resources


class QueriesParameter(parameters.ResourceSelector):
    name = 'queries_resource_id'
    description = 'Queries'
    resource_type = dolbilka_resources.DOLBILKA_STPD_QUERIES
    required = False


class QueriesLimitParameter(parameters.SandboxIntegerParameter):
    name = 'queries_limit'
    description = 'Limit number of queries to read (0 = all)'
    default_value = 0


def fix_default_workers_count(parameters_list):
    for x in parameters_list:
        if x.name == 'workers_count':
            x.default_value = 5
    return parameters_list


class MaxErrorLimitParameter(parameters.SandboxIntegerParameter):
    name = 'max_error_limit'
    description = 'Limit number of errors in responses'
    default_value = 0


class ImagesGetCbirdaemon2Responses(daemons_task.BaseCbirdaemonTask):
    """
        Get cbirdaemon responses
    """

    type = 'IMAGES_GET_CBIRDAEMON2_RESPONSES'

    input_parameters = \
        (QueriesParameter, QueriesLimitParameter, MaxErrorLimitParameter) + \
        daemons_task.BaseCbirdaemonTask.input_parameters + \
        fix_default_workers_count(search_requester_compat.create_params())

    def on_enqueue(self):
        daemons_task.BaseCbirdaemonTask.on_enqueue(self)

        if self._is_apphost_enabled():
            resource_type = daemons_resources.CBIR_DAEMON2_APPHOST_RESPONSES
        else:
            resource_type = daemons_resources.CBIR_DAEMON2_RESPONSES

        self.create_resource(
            self.descr,
            self.__get_responses_path(),
            resource_type
        )

    def on_execute(self):
        cbirdaemon = self._get_daemon()
        # Workaround against different results on different CPUs (CV-318)
        cbirdaemon.get_environment()["MKL_CBWR"] = "SSE4_2"
        with cbirdaemon:
            self.__save_responses(cbirdaemon)

    def _get_queries_parameter(self):
        return QueriesParameter

    def __save_responses(self, component):
        queries_path = self.sync_resource(self.ctx[QueriesParameter.name])
        queries_limit = self.ctx[QueriesLimitParameter.name]
        max_error_limit = self.ctx[MaxErrorLimitParameter.name]
        eh.ensure(os.stat(queries_path).st_size, "Input file with queries is empty")

        queries_iterator = self.__generate_queries(component, queries_path, queries_limit)
        def get_http_error(v):
            for l in v.split("\n"):
                name = l[:l.find("{")]
                value = l[l.find("{")+1:l.rfind("}")]
                if name == "error":
                    return value
            return None
        get_error = get_http_error
        if self._is_apphost_enabled():
            converter_path = self.sync_resource(utils.get_and_check_last_released_resource_id(
                resource_types.APP_HOST_TOOL_CONVERTER_EXECUTABLE,
                arch=sandboxapi.ARCH_LINUX
            ))
            converter_func = apphost_converter.convert_input_response
            def get_apphost_error(v):
                ans = converter_func(converter_path, v)
                if "answers" not in ans:
                    return "Empty answers"
                for a in ans["answers"]:
                    if "signatures" in a:
                        signatures = a["signatures"]
                        if "error" in signatures:
                            return signatures["error"]
                    for s in a.get("signatures_vec", []):
                        if s["name"] == "error":
                            return s["value"]
                return None
            get_error = get_apphost_error
        errors_msg = []
        with open(self.__get_responses_path(), "w") as responses_file:
            for _, status, data in search_requester_compat.save_responses(self.ctx, queries_iterator, url_read=url_read):
                if not status:
                    raise errors.SandboxTaskFailureError("Error during request processing: {}".format(data))
                error = get_error(data)
                if error is not None:
                    errors_msg.append(error)
                if self._is_apphost_enabled():
                    search_requester.write_binary_data(responses_file, data)
                else:
                    responses_file.write("{}\n".format(len(data)))
                    responses_file.write(data)
        if len(errors_msg) > max_error_limit:
            raise errors.SandboxTaskFailureError("Cbirdaemon responses has errors:\n{}".format("\n".join(errors_msg)))
        elif len(errors_msg) > 0:
            logging.info("Cbirdaemon responses has errors:\n{}".format("\n".join(errors_msg)))

    def __get_responses_path(self):
        return self.abs_path("responses.txt")

    def __generate_queries(self, component, queries_path, queries_limit):
        try:
            counter = 1
            with open(queries_path) as queries_file:
                while True:
                    if queries_limit and counter > queries_limit:
                        break

                    header = queries_file.readline()

                    if not header:  # end of file
                        break
                    elif not header.strip():  # empty line
                        continue

                    size = int(header.split(" ")[0])  # header format is '<size> [<timestamp> [<tag>]]'
                    content = queries_file.read(size)
                    if len(content) != size:
                        raise Exception("Invalid input file")
                    counter += 1
                    yield component.port, content
        except Exception as e:
            logging.info("Problem during query generation: {}".format(e))
            yield None


def url_read(args):
    data, timeout, max_retries = args
    if data is None:
        return False, "Problem in queries generator"

    port, raw_request = data

    # reformat raw request for urllib2
    content = StringIO.StringIO(raw_request)
    query = content.readline().split()[1]
    headers = {}
    for header in content:
        header = header.strip()
        if not header:
            break
        header_name, header_value = header.split(":", 1)
        headers[header_name.strip()] = header_value.strip()
    payload = content.read()

    except_msg = ""
    for attempt in xrange(max_retries + 1):
        try:
            request = urllib2.Request("http://localhost:{}{}".format(port, query), payload, headers)
            response = urllib2.urlopen(request, timeout=timeout)
            return True, response.read()
        except Exception as e:
            logging.debug("Problems with request (attempt {}):\n{}".format(attempt, raw_request))
            except_msg = str(e)
    logging.debug("Unresolved problems with request '{}':\n{}".format(except_msg, raw_request))
    return False, except_msg


__Task__ = ImagesGetCbirdaemon2Responses
