import logging
import os.path
import urllib2
import cStringIO as StringIO

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.search import requester as search_requester
from sandbox.projects.common.search import requester_compat as search_requester_compat
from sandbox.projects.common.dolbilka import resources as dolbilka_resources
from sandbox.projects.images.daemons import task as daemons_task
from sandbox.projects.images.daemons import resources as daemons_resources


class QueriesParameter(parameters.ResourceSelector):
    name = 'queries_resource_id'
    description = 'Queries'
    resource_type = dolbilka_resources.DOLBILKA_STPD_QUERIES
    required = False


class QueriesLimitParameter(parameters.SandboxIntegerParameter):
    name = 'queries_limit'
    description = 'Limit number of queries to read (0 = all)'
    default_value = 0


class ImagesGetRimpatchdaemonResponses(daemons_task.BaseRimpatchdaemonTask):
    """
        Get rimpatchdaemon responses
    """

    type = 'IMAGES_GET_RIMPATCHDAEMON_RESPONSES'

    input_parameters = \
        (QueriesParameter, QueriesLimitParameter) + \
        daemons_task.BaseRimpatchdaemonTask.input_parameters + \
        search_requester_compat.create_params()

    def on_enqueue(self):
        daemons_task.BaseRimpatchdaemonTask.on_enqueue(self)

        self.create_resource(
            self.descr,
            self.__get_responses_path(),
            daemons_resources.IMAGES_RIMPATCHDAEMON_APPHOST_RESPONSES
        )

    def on_execute(self):
        rimpatchdaemon = self._get_daemon()
        with rimpatchdaemon:
            self.__save_responses(rimpatchdaemon)

    def _get_queries_parameter(self):
        return QueriesParameter

    def __save_responses(self, component):
        queries_path = self.sync_resource(self.ctx[QueriesParameter.name])
        queries_limit = self.ctx[QueriesLimitParameter.name]
        eh.ensure(os.stat(queries_path).st_size, "Input file with queries is empty")

        queries_iterator = self.__generate_queries(component, queries_path, queries_limit)

        with open(self.__get_responses_path(), "w") as responses_file:
            for _, status, data in search_requester_compat.save_responses(self.ctx, queries_iterator, url_read=url_read):
                if not status:
                    raise errors.SandboxTaskFailureError("Error during request processing: {}".format(data))

                search_requester.write_binary_data(responses_file, data)

    def __get_responses_path(self):
        return self.abs_path("responses.txt")

    def __generate_queries(self, component, queries_path, queries_limit):
        try:
            counter = 1
            with open(queries_path) as queries_file:
                while True:
                    if queries_limit and counter > queries_limit:
                        break

                    header = queries_file.readline()

                    if not header:  # end of file
                        break
                    elif not header.strip():  # empty line
                        continue

                    size = int(header.split(" ")[0])  # header format is '<size> [<timestamp> [<tag>]]'
                    content = queries_file.read(size)
                    if len(content) != size:
                        raise Exception("Invalid input file")
                    counter += 1
                    yield component.port, content
        except Exception as e:
            logging.info("Problem during query generation: {}".format(e))
            yield None


def url_read(args):
    data, timeout, max_retries = args
    if data is None:
        return False, "Problem in queries generator"

    port, raw_request = data

    # reformat raw request for urllib2
    content = StringIO.StringIO(raw_request)
    query = content.readline().split()[1]
    headers = {}
    for header in content:
        header = header.strip()
        if not header:
            break
        header_name, header_value = header.split(":", 1)
        headers[header_name.strip()] = header_value.strip()
    payload = content.read()

    except_msg = ""
    for attempt in xrange(max_retries + 1):
        try:
            request = urllib2.Request("http://localhost:{}{}".format(port, query), payload, headers)
            response = urllib2.urlopen(request, timeout=timeout)
            return True, response.read()
        except Exception as e:
            logging.debug("Problems with request (attempt {}):\n{}".format(attempt, raw_request))
            except_msg = str(e)
    logging.debug("Unresolved problems with request '{}':\n{}".format(except_msg, raw_request))
    return False, except_msg


__Task__ = ImagesGetRimpatchdaemonResponses
