import itertools
import json

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import task

from sandbox.projects.images.daemons import resources as images_daemons_resources


_QUERY = 0
_CODE = 1
_RESPONSE = 2
_BIN_COUNT = 10


class Responses1Parameter(parameters.ResourceSelector):
    name = "rim_daemon_responses1_resource_id"
    description = "RIM responses #1"
    resource_type = images_daemons_resources.IMAGES_RIM_RESPONSES


class Responses2Parameter(parameters.ResourceSelector):
    name = "rim_daemon_responses2_resource_id"
    description = "RIM responses #2"
    resource_type = images_daemons_resources.IMAGES_RIM_RESPONSES


class ResponseId():
    def __init__(self, doc_id, thumb_id, img_url):
        self._doc_id = doc_id
        self._thumb_id = thumb_id
        self._img_url = img_url

    def __eq__(self, other):
        return self._doc_id == other._doc_id or self._thumb_id == other._thumb_id

    def __ne__(self, other):
        return not self.__eq__(other)

    def __str__(self):
        return "{}:{}:{}".format(self._doc_id, self._thumb_id, self._img_url)


# TODO: Make common class to compare results for priemka and TestEnv
class ImagesProdCompareRimResponses(task.SandboxTask):
    """
        Compare rim responses from remote (production) host
    """

    type = 'IMAGES_PROD_COMPARE_RIM_RESPONSES'

    input_parameters = (Responses1Parameter, Responses2Parameter)
    max_fails_percent = 50

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        self.create_resource(
            self.descr,
            self._get_diff(),
            images_daemons_resources.IMAGES_RIM_RESPONSES_DIFF
        )

    def on_execute(self):
        resp_count = 0
        resp_ok_count = 0
        resp1_404 = 0
        resp2_404 = 0
        resp_200 = 0
        dups_distribution = [0] * (_BIN_COUNT + 1)

        iterator1 = self._responses_iterator(self.ctx[Responses1Parameter.name])
        iterator2 = self._responses_iterator(self.ctx[Responses2Parameter.name])

        diff_file = open(self._get_diff(), "w")
        for n, responses in enumerate(itertools.izip_longest(iterator1, iterator2)):
            response1, response2 = responses
            resp_count += 1

            if response1 is None or response2 is None:
                raise errors.SandboxTaskFailureError("Different number of responses")
            if response1[_QUERY] != response2[_QUERY]:
                raise errors.SandboxTaskFailureError("Responses from different plans")

            if response1[_CODE] != "200" and response1[_CODE] != "404":
                continue
            if response2[_CODE] != "200" and response2[_CODE] != "404":
                continue

            resp_ok_count += 1

            if response1[_CODE] == "404":
                resp1_404 += 1
            if response2[_CODE] == "404":
                resp2_404 += 1

            if response1[_CODE] != "200" or response2[_CODE] != "200":
                continue

            resp_200 += 1

            if response1[_RESPONSE] == response2[_RESPONSE]:
                dups_distribution[_BIN_COUNT] += 1
            else:
                dups = [resp for resp in response1[_RESPONSE] if resp in response2[_RESPONSE]]
                response_len = max(len(response1[_RESPONSE]), len(response2[_RESPONSE]))
                dups_fraction = len(dups) / float(response_len) if response_len > 0 else 0.0
                dups_fraction = int(dups_fraction * _BIN_COUNT)

                if len(dups) > 0:
                    dups_distribution[dups_fraction] += 1

                diff = ["#{}: {} != {}".format(i, str(pair[0]), str(pair[1])) for i, pair in enumerate(itertools.izip_longest(response1[_RESPONSE], response2[_RESPONSE])) if pair[0] is None or pair[1] is None or pair[0] != pair[1]]
                diff_file.write(response1[_QUERY] + "\t" + "\t".join(diff) + "\n")

        diff_file.close()

        resp_sum = 0
        for i, cnt in reversed(list(enumerate(dups_distribution))):
            dups_distribution[i] = 100 * float(cnt + resp_sum) / resp_200
            resp_sum += cnt

        if resp_count == 0 or 100 * (resp_count - resp_ok_count) / resp_count > self.max_fails_percent:
            raise errors.SandboxTaskFailureError("It's not comparable. No responses or fail response count exceeds {}%".format(self.max_fails_percent))

        results = {}
        results["resp_count"] = resp_count
        results["resp_ok"] = resp_ok_count
        results["resp1_404"] = resp1_404
        results["resp2_404"] = resp2_404
        results["resp_200"] = resp_200
        results["dups_distr"] = dups_distribution
        self.ctx["results"] = results

    def _responses_iterator(self, responses_resource_id):
        responses_path = self.sync_resource(responses_resource_id)
        with open(responses_path) as responses_file:
            for line in responses_file:
                parts = line.rstrip('\n').split('\t')
                if parts[_RESPONSE]:
                    json_response = json.loads(parts[_RESPONSE])
                    parts[_RESPONSE] = [ResponseId(doc["id"], doc["tid"], doc["s"][0]["iu"]) for doc in json_response["rld"]]
                else:
                    parts[_RESPONSE] = []
                yield parts

    def _get_diff(self):
        return self.abs_path("responses.diff")


__Task__ = ImagesProdCompareRimResponses
