from sandbox.projects.common.mediasearch.online import control, command
import itertools as itt
import requests


class ImagesProdFetchRimResponses(control.BaseControl):
    default_query_url_template = "https://imgs-rim.yandex.net%s"
    default_queries_limit = 5000
    default_ordered_output = True
    default_build_output = True

    @staticmethod
    def iterate_queries(plan, queries_count):
        with open(plan) as f:
            for line in itt.islice(f, queries_count):
                yield line.rstrip()

    def create_query_url(self, query):
        url_template = self.query_url_template.replace("%s", "{query}")
        return url_template.format(query=query)

    def process_query(self, query):
        query_url = self.create_query_url(query)
        resp = None
        try:
            resp = self.fetch(query_url, on_404="resp_404", on_error="resp_error")
        except requests.HTTPError as e:
            self.output += query + "\t" + str(e.response.status_code) + "\t\n"
            raise
        except Exception as e:
            self.output += query + "\t0\t\n"
            raise

        self.counter.inc("resp_ok")
        self.output += query + "\t" + "200" + "\t" + resp + "\n"

    @staticmethod
    def template_data(res):
        data = []
        total_queries = res.get("total_queries", 0)
        data.append({"name": "Total queries", "value": str(total_queries), "fraction": None})

        if total_queries > 0:
            resp_good = res.get("resp_ok", 0)
            data.append({"name": "Good responses",
                         "value": str(resp_good),
                         "fraction": float(resp_good) / total_queries})

            resp_404 = res.get("resp_404", 0)
            data.append({"name": "404 responses",
                         "value": str(resp_404),
                         "fraction": float(resp_404) / total_queries})

            resp_bad = res.get("resp_error", 0)
            data.append({"name": "Error responses",
                         "value": str(resp_bad),
                         "fraction": float(resp_bad) / total_queries})

        return data


if __name__ == '__main__':
    command.command(ImagesProdFetchRimResponses)
