import io
import json
import urlparse

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import task

from sandbox.projects.images.daemons import resources as images_daemons_resources


_QUERY = 0
_CODE = 1
_RESPONSE = 2


class ResponsesParameter(parameters.ResourceSelector):
    name = "rim_daemon_responses_resource_id"
    description = "RIM responses"
    resource_type = images_daemons_resources.IMAGES_RIM_RESPONSES


class ImagesProdParseRimResponses(task.SandboxTask):
    """
        Parse rim responses to tsv
    """

    type = 'IMAGES_PROD_PARSE_RIM_RESPONSES'

    input_parameters = (ResponsesParameter,)

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        self.create_resource(
            self.descr,
            self._get_output_urls(),
            images_daemons_resources.IMAGES_RIM_RESPONSES_URLS
        )
        self.create_resource(
            self.descr,
            self._get_output_dups(),
            images_daemons_resources.IMAGES_RIM_RESPONSES_DUP_URLS
        )

    def on_execute(self):
        with io.open(self._get_output_urls(), "w", encoding='utf8') as urls_file:
            with io.open(self._get_output_dups(), "w", encoding='utf8') as dups_file:
                for response in self._responses_iterator(self.ctx[ResponsesParameter.name]):
                    if response[_CODE] != "200":
                        continue
                    url_parts = urlparse.urlparse(response[_QUERY])
                    src_urls = urlparse.parse_qs(url_parts.query)['img_url']
                    for related_urls in response[_RESPONSE]:
                        urls_file.write(u"{}\t{}\n".format(src_urls[0], related_urls[0]))
                        dups_file.write(json.dumps({"source_urls": src_urls, "related_urls": related_urls}).decode('utf8') + u"\n")

    def _responses_iterator(self, responses_resource_id):
        responses_path = self.sync_resource(responses_resource_id)
        with io.open(responses_path, encoding='utf8') as responses_file:
            for line in responses_file:
                parts = line.rstrip('\n').split('\t')
                if parts[_RESPONSE]:
                    json_response = json.loads(parts[_RESPONSE])
                    parts[_RESPONSE] = [[snip["iu"] for snip in doc["s"]] for doc in json_response["rld"]]
                else:
                    parts[_RESPONSE] = []
                yield parts

    def _get_output_urls(self):
        return self.abs_path("responses.urls")

    def _get_output_dups(self):
        return self.abs_path("responses.dup.urls")


__Task__ = ImagesProdParseRimResponses
