import httplib
import base64

from datetime import datetime

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import network
from sandbox.sandboxsdk import parameters

from sandbox.projects import resource_types
from sandbox.projects.common.thumbdaemon import utils as thumbdaemon_utils
from sandbox.projects.images.daemons import task as daemons_task


_DATE_HEADER = 'Date'


class RFC1123:
    str_format = "%a, %d %b %Y %H:%M:%S %Z"

    @staticmethod
    def str2datetime(value):
        return datetime.strptime(value, RFC1123.str_format)

    @staticmethod
    def datetime2str(value):
        # Time zone is not set by default
        return value.strftime(RFC1123.str_format) + 'GMT'


# Note: this is a fake resource for database autoselection in TestEnvironment
class QueriesParameter(parameters.ResourceSelector):
    name = 'queries_resource_id'
    description = 'Queries'
    resource_type = resource_types.PLAIN_TEXT_QUERIES
    required = True


def check_header(name, expected_value, headers):
    value = headers.get(name.lower(), None)
    if value != expected_value:
        raise errors.SandboxTaskFailureError("Header {}: expected {}, got {}".format(name, expected_value, value))


class ImagesTestNaildaemonFunctional(daemons_task.BaseNaildaemonTask):
    """
        Test basic functionality of nail daemon
    """
    type = 'IMAGES_TEST_NAILDAEMON_FUNCTIONAL'

    input_parameters = (QueriesParameter,) + daemons_task.BaseNaildaemonTask.input_parameters

    def on_execute(self):
        database_resource_id = self.ctx[daemons_task.NAILDAEMON_PARAMS.Database.name]
        thumb_id = next(thumbdaemon_utils.get_thumb_ids(database_resource_id))
        shard_type, shard_num, shard_timestamp = thumbdaemon_utils.get_database_version(database_resource_id)

        thumb_daemon = self._get_daemon()
        thumb_daemon.start()
        thumb_daemon.wait()

        test_http_request(thumb_daemon.port, '/version', httplib.OK, shard_timestamp)
        test_http_request(thumb_daemon.port, '/shardname', httplib.OK, "-".join((shard_type, shard_num, shard_timestamp)))
        test_http_request(thumb_daemon.port, '/i?id=crazyid', httplib.NOT_FOUND)
        test_http_request(thumb_daemon.port, '/crazypath', httplib.NOT_FOUND)
        test_http_request(thumb_daemon.port, '/robots.txt', httplib.OK, "User-Agent: *\nDisallow: /\n")

        test_http_request(thumb_daemon.port, '/i?id={}&n=crazy'.format(thumb_id), httplib.OK)

        content_type_name = 'Content-Type'
        content_length_name = 'Content-Length'
        timing_allow_origin_name = 'Timing-Allow-Origin'
        thumb_path = '/i?id={}'.format(thumb_id)
        body, headers = test_http_request(thumb_daemon.port, thumb_path, httplib.OK)

        check_header(content_type_name, 'image/jpeg', headers)
        check_header(content_length_name, str(len(body)), headers)
        check_header(timing_allow_origin_name, '*', headers)

        base64_body, base64_headers = test_http_request(thumb_daemon.port, thumb_path + '&b64=1', httplib.OK)

        check_header(content_type_name, 'text/plain', base64_headers)
        check_header(content_length_name, str(len(base64_body)), base64_headers)
        check_header(timing_allow_origin_name, '*', base64_headers)

        expected_base64_body = 'data:image/jpeg;base64,' + base64.b64encode(body)
        if base64_body != expected_base64_body:
            raise errors.SandboxTaskFailureError("Base64 body: expected {}, got {}".format(expected_base64_body, base64_body))

        date_value = headers.get(_DATE_HEADER.lower(), None)
        utcnow = datetime.utcnow()
        if not dates_are_near(RFC1123.str2datetime(date_value), utcnow):
            raise errors.SandboxTaskFailureError("Date value {} different from now {}".format(date_value, utcnow))

        # test 304 support
        etag_name = 'ETag'
        if_none_match_name = 'If-None-Match'
        etag_value = headers.get(etag_name.lower(), None)
        if etag_value is None:
            raise errors.SandboxTaskFailureError("Did not get {} header".format(etag_name))
        # When an invalid etag comes in If-None-Match, ignore it and serve full response
        test_http_request(thumb_daemon.port, thumb_path, httplib.OK, body, headers={
            if_none_match_name: etag_value[:-2] + 'x"',
        })
        _, headers304 = test_http_request(thumb_daemon.port, thumb_path, httplib.NOT_MODIFIED, '', headers={
            if_none_match_name: etag_value,
        })
        for name in [content_type_name]:
            if name.lower() in headers304:
                raise errors.SandboxTaskFailureError('Unexpected header {} in Not Modified response'.format(name))

        # shutdown it manually, no .stop() required
        test_http_request_one_method(thumb_daemon.port, 'GET', '/admin?action=shutdown', httplib.OK)

        shutdown_timeout = 3
        if not network.wait_port_is_free(port=thumb_daemon.port, timeout=shutdown_timeout):
            raise errors.SandboxTaskFailureError("Could not shut down thumbdaemon in {} seconds".format(shutdown_timeout))

    def _get_queries_parameter(self):
        return QueriesParameter


def test_http_request_one_method(port, method, http_path, expected_code, expected_data=None, headers={}):
    try:
        connection = httplib.HTTPConnection('localhost', port)
        connection.request(method, http_path, headers=headers)
        response = connection.getresponse()
    except (IOError, httplib.HTTPException) as e:
        raise errors.SandboxTaskFailureError("Request '{}', failed to retrieve response from server: {}".format(
                                             http_path, e))

    if expected_code != response.status:
        raise errors.SandboxTaskFailureError("Request '{}', method {}, expected code {}, got {}".format(
                                             http_path, method, expected_code, response.status))
    data = response.read()
    if expected_data is not None:
        if data != expected_data:
            raise errors.SandboxTaskFailureError("Request '{}', expected data\n'{}'\ngot\n'{}'\n".format(http_path, expected_data, data))

    return data, dict([(x[0].lower(), x[1]) for x in response.getheaders()])


def test_http_request(port, http_path, expected_code, expected_data=None, headers={}):
    _, head_headers = test_http_request_one_method(port, 'HEAD', http_path, expected_code, '', headers)
    data, get_headers = test_http_request_one_method(port, 'GET', http_path, expected_code, expected_data, headers)
    head_date = head_headers[_DATE_HEADER.lower()]
    get_date = get_headers[_DATE_HEADER.lower()]
    if not dates_are_near(RFC1123.str2datetime(head_date), RFC1123.str2datetime(get_date)):
        raise errors.SandboxTaskFailureError("HEAD and GET dates {}, {} are not near".format(head_date, get_date))
    # To make comparing headers easier
    get_headers[_DATE_HEADER.lower()] = head_headers[_DATE_HEADER.lower()]
    if head_headers != get_headers:
        raise errors.SandboxTaskFailureError("Request {}, headers for HEAD and GET are different:\n{}\n{}\n".format(http_path, head_headers, get_headers))

    return data, get_headers


def dates_are_near(date1, date2):
    return abs((date1 - date2).total_seconds()) < 5


__Task__ = ImagesTestNaildaemonFunctional
