import json
import logging
import os
import tarfile

from sandbox import common
import sandbox.common.types.misc as ctm

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common import cgroup as cgroup_api
from sandbox.projects.common.search import components as search_components
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.common.search.components import component as components_common
from sandbox.projects.common.thumbdaemon import utils as thumbdaemon_utils

from . import resources as daemons_resources


_RIM_DBPATH_PARAM = 'DbPath:'


class Daemon(components_common.ProcessComponentMixinWithShutdown,
             components_common.WaitPortComponentMixin,
             components_common.Component):
    """Basic class for all new style components"""

    def __init__(self, args, start_timeout=60, cgroup=None):
        self._port = components_common.try_get_free_port()
        self.__name = os.path.basename(args[0])
        self.__cgroup = cgroup
        logging.info("Using port {} for {}".format(self._port, self.__name))

        components_common.ProcessComponentMixinWithShutdown.__init__(
            self,
            args=[str(a).format(port=self._port) for a in args],
            shutdown_url="http://localhost:{}/admin?action=shutdown".format(self._port),
            log_prefix=self.__name,
            preexec_fn=lambda: self.__cgroup.set_current() if self.__cgroup else None
        )
        components_common.WaitPortComponentMixin.__init__(
            self,
            endpoints=[("localhost", self._port)],
            wait_timeout=start_timeout
        )

    def set_cgroup(self, cgroup):
        """
            Change component's cgroup

            Used by sanitizer library to reset cgroup
        """
        self.__cgroup = cgroup

    def replace_config_parameter(self, *args, **kwargs):
        """
            Useless method called by sanitizer library
        """
        pass

    @property
    def port(self):
        return int(self._port)


class Rimdaemon(Daemon):
    """
        New experimental component based on common mixins
    """

    def __init__(self,
                 binary_path,
                 config_path,
                 database_path,
                 start_timeout=search_components.DEFAULT_START_TIMEOUT,
                 shutdown_timeout=120,
                 cgroup=None):

        database_path = os.path.join(database_path, self.__get_database_prefix(config_path))
        args = [
            binary_path,
            '--port', '{port}',
            '--config', config_path,
            '--rim-db', database_path,
            '--threads', str(20)
        ]
        Daemon.__init__(self, args, cgroup=cgroup, start_timeout=start_timeout)

    def __get_database_prefix(self, config_path):
        with open(config_path, "r") as f:
            for line in f:
                if line.startswith(_RIM_DBPATH_PARAM):
                    return line[7:].strip().strip('"').split('/')[-1]
        return 'rimdb'


def create_rimdaemon_params(n='', group_name=None, component_name=None):
    if group_name is None:
        group_name = "Rimdaemon{} parameters".format(n)

    if component_name is None:
        component_name = "rimdaemon{}".format(n)

    class Parameters:
        class Binary(parameters.ResourceSelector):
            name = '{}_executable_resource_id'.format(component_name)
            description = 'Rimdaemon executable:'
            resource_type = daemons_resources.IMAGES_RIM_DAEMON_EXECUTABLE
            group = group_name
            required = True

        class Config(parameters.ResourceSelector):
            name = '{}_config_resource_id'.format(component_name)
            description = 'Rimdaemon config:'
            resource_type = daemons_resources.IMAGES_RIM_DAEMON_CONFIG
            group = group_name
            required = True

        class Database(parameters.ResourceSelector):
            name = '{}_database_resource_id'.format(component_name)
            description = 'Rimdaemon DB:'
            resource_type = resource_types.IMAGES_SEARCH_DATABASE
            group = group_name
            required = True

        class StartTimeout(parameters.SandboxIntegerParameter):
            name = '{}_start_timeout'.format(component_name)
            description = 'Start timeout (sec)'
            group = group_name
            default_value = search_components.DEFAULT_START_TIMEOUT

        class Cgroup(parameters.SandboxStringParameter):
            name = '{}_cgroup'.format(component_name)
            description = 'Cgroup'
            group = group_name

        params = (Binary, Config, Database, StartTimeout, Cgroup)

    return Parameters


def get_rimdaemon(params=create_rimdaemon_params(), **kwargs):
    task = channel.task
    cgroup = _create_basesearch_cgroup(
        search_settings.INDEX_RIM,
        utils.get_or_default(task.ctx, params.Cgroup)
    )

    return Rimdaemon(
        binary_path=task.sync_resource(task.ctx[params.Binary.name]),
        config_path=task.sync_resource(task.ctx[params.Config.name]),
        database_path=task.sync_resource(task.ctx[params.Database.name]),
        start_timeout=utils.get_or_default(task.ctx, params.StartTimeout),
        cgroup=cgroup,
        **kwargs
    )


class Rimpatchdaemon(Daemon):
    """
        RIM Patch daemon wrapper
    """

    def __init__(self,
                 binary_path,
                 config_path,
                 start_timeout=search_components.DEFAULT_START_TIMEOUT,
                 shutdown_timeout=120,
                 cgroup=None):

        args = [
            binary_path,
            '--port', '{port}',
            '--config', config_path,
            '--threads', str(20)
        ]
        Daemon.__init__(self, args, cgroup=cgroup, start_timeout=start_timeout)

    @property
    def port(self):
        return int(self._port) + 1


def create_rimpatchdaemon_params(n='', group_name=None, component_name=None):
    if group_name is None:
        group_name = "Rimdaemon Patch {} parameters".format(n)

    if component_name is None:
        component_name = "rimpatchdaemon{}".format(n)

    class Parameters:
        class Binary(parameters.ResourceSelector):
            name = '{}_executable_resource_id'.format(component_name)
            description = 'Rimpatchdaemon executable:'
            resource_type = daemons_resources.IMAGES_RIMPATCH_DAEMON_EXECUTABLE
            group = group_name
            required = True

        class Config(parameters.ResourceSelector):
            name = '{}_config_resource_id'.format(component_name)
            description = 'Rimpatchdaemon config:'
            resource_type = daemons_resources.IMAGES_RIMPATCH_DAEMON_CONFIG
            group = group_name
            required = True

        class BanArchive(parameters.ResourceSelector):
            name = '{}_database_resource_id'.format(component_name)
            description = 'Archive with ban files:'
            resource_type = daemons_resources.IMAGES_RIMPATCH_DAEMON_BAN_ARCHIVE
            group = group_name
            required = True

        class StartTimeout(parameters.SandboxIntegerParameter):
            name = '{}_start_timeout'.format(component_name)
            description = 'Start timeout (sec)'
            group = group_name
            default_value = search_components.DEFAULT_START_TIMEOUT

        class Cgroup(parameters.SandboxStringParameter):
            name = '{}_cgroup'.format(component_name)
            description = 'Cgroup'
            group = group_name

        params = (Binary, Config, BanArchive, StartTimeout, Cgroup)

    return Parameters


def get_rimpatchdaemon(params=create_rimpatchdaemon_params(), **kwargs):
    task = channel.task
    cgroup = _create_basesearch_cgroup(
        search_settings.INDEX_RIMPATCH,
        utils.get_or_default(task.ctx, params.Cgroup)
    )

    ban_archive = task.sync_resource(task.ctx[params.BanArchive.name])
    with tarfile.open(ban_archive) as tar:
        tar.extractall()

    return Rimpatchdaemon(
        binary_path=task.sync_resource(task.ctx[params.Binary.name]),
        config_path=task.sync_resource(task.ctx[params.Config.name]),
        start_timeout=utils.get_or_default(task.ctx, params.StartTimeout),
        cgroup=cgroup,
        **kwargs
    )


class Cbirdaemon(Daemon):

    def __init__(self,
                 binary_path,
                 config_path,
                 settings_path,
                 main_threads=0,
                 apphost_threads=0,
                 danet_threads=0,
                 start_timeout=search_components.DEFAULT_START_TIMEOUT,
                 shutdown_timeout=120,
                 cgroup=None):
        args = [
            binary_path,
            '--port', '{port}',
            '--config', config_path,
            '--data-dir', settings_path,
        ]

        if main_threads:
            args.extend(["--threads", main_threads])

        if apphost_threads:
            args.extend(["--apphost-threads", apphost_threads])

        if danet_threads:
            args.extend(["--danet-threads", danet_threads])

        Daemon.__init__(self, args, cgroup=cgroup, start_timeout=start_timeout)


class CbirdaemonServant(components_common.Component):
    """
    Wrapper over cbirdaemon2 component to obtain apphost port
    """

    def __init__(self, component):
        self.component = component

    def start(self):
        self.component.start()

    def wait(self, *args, **kwargs):
        self.component.wait(*args, **kwargs)

    def stop(self, *args, **kwargs):
        self.component.stop(*args, **kwargs)

    @property
    def port(self):
        return int(self.component.port) + 1

    @property
    def process(self):
        return self.component.process

    def get_environment(self):
        return self.component.get_environment()

    def set_environment(self, value):
        return self.component.set_environment(value)

    @property
    def run_cmd_patcher(self):
        return self.component.run_cmd_patcher

    @run_cmd_patcher.setter
    def run_cmd_patcher(self, value):
        self.component.run_cmd_patcher = value


def create_cbirdaemon_params(n="", group_name=None, component_name=None):
    if group_name is None:
        group_name = "Cbirdaemon{} parameters".format(n)

    if component_name is None:
        component_name = "cbir_daemon{}".format(n)

    class Parameters:
        class Binary(parameters.ResourceSelector):
            name = "{}_executable_resource_id".format(component_name)
            description = 'Executable'
            resource_type = (daemons_resources.CBIR_DAEMON2_EXECUTABLE,
                             daemons_resources.CBIR_DAEMON2_OMP_EXECUTABLE,
                             daemons_resources.CBIR_DAEMON2_GPU_EXECUTABLE)
            group = group_name
            required = True

        class Config(parameters.ResourceSelector):
            name = "{}_config_resource_id".format(component_name)
            description = 'Config'
            resource_type = (daemons_resources.CBIR_DAEMON2_CONFIG,
                             daemons_resources.CBIR_DAEMON2_OMP_CONFIG,
                             daemons_resources.CBIR_DAEMON2_GPU_CONFIG,
                             daemons_resources.CBIR_DAEMON2_API_CONFIG,
                             daemons_resources.CBIR_DAEMON2_API_GPU_CONFIG)
            group = group_name
            required = True

        class Settings(parameters.ResourceSelector):
            name = "{}_settings_resource_id".format(component_name)
            description = 'Settings'
            resource_type = (daemons_resources.CBIR_DAEMON2_SETTINGS,
                             daemons_resources.CBIR_DAEMON2_API_SETTINGS,
                             daemons_resources.CBIR_DAEMON2_GPU_SETTINGS,
                             daemons_resources.CBIR_DAEMON2_GPU_API_SETTINGS)
            group = group_name
            required = False

        class MainThreads(parameters.SandboxIntegerParameter):
            name = "{}_main_threads".format(component_name)
            description = 'Number of main threads (use 0 for default)'
            group = group_name
            default_value = 0

        class ApphostThreads(parameters.SandboxIntegerParameter):
            name = "{}_apphost_threads".format(component_name)
            description = 'Number of apphost threads (use 0 for default)'
            group = group_name
            default_value = 0

        class DanetThreads(parameters.SandboxIntegerParameter):
            name = "{}_danet_threads".format(component_name)
            description = 'Number of danet threads (use 0 for default)'
            group = group_name
            default_value = 0

        class StartTimeout(parameters.SandboxIntegerParameter):
            name = "{}_start_timeout".format(component_name)
            description = 'Start timeout (sec)'
            group = group_name
            default_value = search_components.DEFAULT_START_TIMEOUT

        class EnableAppHost(parameters.SandboxBoolParameter):
            name = "{}_enable_app_host".format(component_name)
            description = 'Enable AppHost'
            group = group_name
            default_value = False

        class Cgroup(parameters.SandboxStringParameter):
            name = '{}_cgroup'.format(component_name)
            description = 'Cgroup'
            group = group_name

        params = (
            Binary,
            Config,
            Settings,
            MainThreads,
            ApphostThreads,
            DanetThreads,
            Cgroup,
            StartTimeout,
            EnableAppHost
        )

    return Parameters


def get_cbirdaemon(params=create_cbirdaemon_params(), **kwargs):
    task = channel.task
    cgroup = _create_basesearch_cgroup(
        search_settings.INDEX_CBIR_DAEMON,
        utils.get_or_default(task.ctx, params.Cgroup)
    )

    # Workaround for backward compatibility
    settings_resource_id = utils.get_or_default(task.ctx, params.Settings)
    if settings_resource_id:
        settings_path = task.sync_resource(settings_resource_id)
    else:
        settings_path = None

    component = Cbirdaemon(
        binary_path=task.sync_resource(task.ctx[params.Binary.name]),
        config_path=task.sync_resource(task.ctx[params.Config.name]),
        settings_path=settings_path,
        start_timeout=utils.get_or_default(task.ctx, params.StartTimeout),
        cgroup=cgroup,
        main_threads=utils.get_or_default(task.ctx, params.MainThreads),
        apphost_threads=utils.get_or_default(task.ctx, params.ApphostThreads),
        danet_threads=utils.get_or_default(task.ctx, params.DanetThreads),
        **kwargs
    )

    if utils.get_or_default(task.ctx, params.EnableAppHost):
        component = CbirdaemonServant(component)

    return component


class Naildaemon(Daemon):
    """
        New experimental component based on common mixins
    """

    def __init__(self,
                 binary_path,
                 config_path,
                 database_path,
                 start_timeout=search_components.DEFAULT_START_TIMEOUT,
                 shutdown_timeout=120,
                 cgroup=None):

        args = [
            binary_path,
            "-p", "{port}",
            "-C", config_path,
            "-m", thumbdaemon_utils.thdb_path(database_path),
        ]
        Daemon.__init__(self, args, cgroup=cgroup, start_timeout=start_timeout)


def create_naildaemon_params(n="", group_name=None, component_name=None):
    if group_name is None:
        group_name = "Thumbdaemon{} parameters".format(n)

    if component_name is None:
        component_name = "thumb_daemon{}".format(n)

    class Parameters:
        class Binary(parameters.ResourceSelector):
            name = "{}_executable_resource_id".format(component_name)
            description = 'Executable'
            resource_type = daemons_resources.NAIL_DAEMON_EXECUTABLE
            group = group_name
            required = True

        class Config(parameters.ResourceSelector):
            name = "{}_config_resource_id".format(component_name)
            description = 'Config'
            resource_type = daemons_resources.NAIL_DAEMON_CONFIG
            group = group_name
            required = True

        class Database(parameters.ResourceSelector):
            name = "{}_database_resource_id".format(component_name)
            description = 'Database (index name should be "thdb")'
            resource_type = resource_types.IMAGES_SEARCH_DATABASE
            group = group_name
            required = False

        class StartTimeout(parameters.SandboxIntegerParameter):
            name = "{}_start_timeout".format(component_name)
            description = 'Start timeout (sec)'
            group = group_name
            default_value = search_components.DEFAULT_START_TIMEOUT

        class Cgroup(parameters.SandboxStringParameter):
            name = '{}_cgroup'.format(component_name)
            description = 'Cgroup'
            group = group_name

        params = (Binary, Config, Database, StartTimeout, Cgroup)

    return Parameters


def get_naildaemon(params=create_naildaemon_params(), **kwargs):
    task = channel.task
    cgroup = _create_basesearch_cgroup(
        search_settings.INDEX_THUMB_QUICK,
        utils.get_or_default(task.ctx, params.Cgroup)
    )

    return Naildaemon(
        binary_path=task.sync_resource(task.ctx[params.Binary.name]),
        config_path=task.sync_resource(task.ctx[params.Config.name]),
        database_path=task.sync_resource(task.ctx[params.Database.name]),
        start_timeout=utils.get_or_default(task.ctx, params.StartTimeout),
        cgroup=cgroup,
        **kwargs
    )


def _create_basesearch_cgroup(index_type, cgroup_props):
    if common.config.Registry().common.installation == ctm.Installation.LOCAL:
        logging.info("Skipping cgroups creation on local sandbox")
        return None

    if cgroup_props:
        cgroup_props = json.loads(cgroup_props)
    else:
        cgroup_props = search_settings.ImagesSettings.CGROUPS[index_type]

    return cgroup_api.create_cgroup(index_type, cgroup_props)
