# -*- coding: utf-8 -*-

import logging
import os
import subprocess
import sys

from sandbox.common.types.task import ReleaseStatus

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import ListRepeater, SandboxBoolParameter, SandboxIntegerParameter, SandboxStringParameter
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects import DeployNannyDashboard, MediaTaskMonitoring, resource_types
from sandbox.projects.common import utils
from sandbox.projects.common.nanny import nanny
from sandbox.projects.MediaLib.MediaStoreShardmap import MediaStoreShardmap, ShardmapType, IndexState
from sandbox.projects.MediaLib.iss_shardtracker import iss_shardtracker_api


logger = logging.getLogger()


class PrimaryShardsCount(SandboxStringParameter):
    name = 'primary_shards_count'
    description = 'Shards count'
    required = True
    default_value = '25'


class MaxShardSizeGB(SandboxIntegerParameter):
    name = 'max_shard_size_gb'
    description = 'Max shard size in gigabytes'
    required = False


class MinShardSizeGB(SandboxIntegerParameter):
    name = 'min_shard_size_gb'
    description = 'Min shard size in gigabytes'
    required = False

class MonitServerHost(SandboxStringParameter):
    name = 'monit_server_host'
    description = 'Monit server'
    required = False
    default_value = 'monit.n.yandex-team.ru'


class ShardSizeMetricPrefix(SandboxStringParameter):
    name = 'shard_size_metric_prefix'
    description = 'Shard size metric prefix'
    required = False
    default_value = ''


class SwitchDashboardName(SandboxStringParameter):
    name = 'switch_dashboard_name'
    description = 'Switch Dashboard name'
    required = True
    default_value = 'images-commercial'
    group = 'switch'


class SwitchDashboardRecipe(SandboxStringParameter):
    name = 'switch_dashboard_recipe'
    description = 'Switch Dashboard recipe'
    required = True
    default_value = 'db_switch_commercial'
    group = 'switch'


class SwitchDashboardGroups(SandboxStringParameter):
    name = 'switch_dashboard_groups'
    description = 'Switch Dashboard groups (comma separated)'
    required = False
    default_value = 'daemons,daemons_hamster'
    group = 'switch'


class SwitchDashboardItsTask(ListRepeater, SandboxStringParameter):
    name = 'switch_dashboard_its_task'
    description = 'ITS ручки для обновления версии базы'
    required = False
    default_value = [
    ]
    group = 'switch'


class SwitchDashboardSetZkTask(ListRepeater, SandboxStringParameter):
    name = 'switch_dashboard_set_zk_task'
    description = 'ZK ноды для обновления версии базы'
    required = False
    default_value = [
        '/media-services/images/flags/production_commercial/on_production',
    ]
    group = 'switch'


class MonitoringEnable(SandboxBoolParameter):
    name = 'monitoring_enable'
    description = 'Enable monitoring'
    default_value = True
    group = 'monitoring'


class MonitoringTelegramChatId(MediaTaskMonitoring.TelegramChatId):
    group = 'monitoring'
    default_value = '-1001088652476'


class MonitoringEmail(MediaTaskMonitoring.Email):
    group = 'monitoring'
    default_value = 'images-newdb@yandex-team.ru'


class MonitoringSwitchTime(MediaTaskMonitoring.MonitoringTime):
    name = 'monitoring_switch_time'
    group = 'monitoring'
    description = 'Time to alert on switching stage (seconds)'
    default_value = 3 * 60 * 60  # 3 hours


class ImagesCommercialStoreShardmap(MediaStoreShardmap):
    """Switch Images commercial database"""

    type = 'IMAGES_COMMERCIAL_STORE_SHARDMAP'
    input_parameters = (ShardmapType,
                        PrimaryShardsCount,
                        MaxShardSizeGB,
                        MinShardSizeGB,
                        MonitServerHost,
                        ShardSizeMetricPrefix,
                        IndexState,
                        SwitchDashboardName,
                        SwitchDashboardRecipe,
                        SwitchDashboardGroups,
                        SwitchDashboardSetZkTask,
                        MonitoringEnable,
                        MonitoringTelegramChatId,
                        MonitoringEmail,
                        MonitoringSwitchTime)

    # task to release for shard
    SHARDMAP_RESOURCE = 'IMAGES_COMMERCIAL_SHARDMAP'

    def _run_monitoring_task(self, monitoring_id, monitoring_time):
        task_params = {
            MediaTaskMonitoring.MonitoringTaskId.name: monitoring_id,
            MediaTaskMonitoring.MonitoringTime.name: monitoring_time,
            MediaTaskMonitoring.TelegramChatId.name: self.ctx[MonitoringTelegramChatId.name],
            MediaTaskMonitoring.Email.name: self.ctx[MonitoringEmail.name],
        }
        return SandboxTask.create_subtask(
            self,
            task_type=MediaTaskMonitoring.MediaTaskMonitoring.type,
            description="Monitoring task {}".format(monitoring_id),
            input_parameters=task_params,
            inherit_notifications=True
        ).id

    def _check_shard_size(self):
        shards_count = int(self.ctx[PrimaryShardsCount.name])
        state = self.ctx[IndexState.name]
        limit = self.ctx[MaxShardSizeGB.name]
        min_size_limit = self.ctx[MinShardSizeGB.name]
        shard_sizes = []

        with iss_shardtracker_api() as shardtracker:
            for shard in xrange(shards_count):
                shard_name = self._generate_shard_name(shard, state).split(' ')[1]
                shard_info = shardtracker.getShardInfo(shard_name)
                shard_size = float(shard_info.shardInfo.full.size) / 2 ** 30
                shard_sizes.append(shard_size)

        max_shard_size = max(shard_sizes)
        min_shard_size = min(shard_sizes)
        avg_shard_size = sum(shard_sizes) / shards_count
        info = 'Shards count: {shards_count}\n' \
               'Shard sizes(min/avg/max): {min:.2f}/{avg:.2f}/{max:.2f}GB\n' \
               'Total size: {total_size:.2f}GB'.format(min=min_shard_size,
                                                       avg=avg_shard_size,
                                                       max=max_shard_size,
                                                       shards_count=shards_count,
                                                       total_size=sum(shard_sizes))

        self.set_info(info)
        if max_shard_size > limit:
            raise SandboxTaskFailureError(
                'Shard is too big: {0:.2f}GB with {1:.2f}GB limit\n'.format(max_shard_size, limit))

        if min_shard_size < min_size_limit:
            raise SandboxTaskFailureError(
                'Shard is too small: {0:.2f}GB with {1:.2f}GB min limit\n'.format(min_shard_size, min_size_limit))

        prefix = self.ctx.get('shard_size_metric_prefix')
        monitoring_client_tool_id = utils.get_and_check_last_released_resource_id(
            resource_types.MONITORING_CLIENT_EXECUTABLE
        )
        monitoring_client_tool = utils.sync_resource(monitoring_client_tool_id)
        monitoring_args = [
            monitoring_client_tool,
            'report-metric',
            '--monitoring-server',
            self.ctx.get('monit_server_host'),
        ]
        for metric, value in [['max-shard-size', max_shard_size],
                              ['min-shard-size', min_shard_size],
                              ['avg-shard-size', avg_shard_size],
                              ['shard-size-limit', limit]]:
            try:
                subprocess.check_call(
                    monitoring_args + [
                        '--id', '{}{}'.format(prefix, metric),
                        '--policy', 'abs',
                        '--value', str(value),
                    ]
                )
            except Exception as e:
                logging.error(
                    "Couldn't send metric {}. Reason: {}".format(metric, e.message)
                )

    def on_release(self, additional_parameters):
        shardmap_filename = self._generate_shardmap_filename()

        # Switch DB Logic (STABLE release)
        # ================================
        if additional_parameters['release_status'] == ReleaseStatus.STABLE:
            if self.ctx.get('switch_task_id') is not None:
                raise SandboxTaskFailureError("Task's already been in STABLE")

            if self.ctx[MaxShardSizeGB.name]:
                self._check_shard_size()

            its_values_params = None
            if self.ctx.get(SwitchDashboardItsTask.name):
                its_values_params = {key: self.ctx[IndexState.name] for key in self.ctx[SwitchDashboardItsTask.name]}
            set_zk_values_params = None
            if self.ctx.get(SwitchDashboardSetZkTask.name):
                set_zk_values_params = {key: shardmap_filename for key in self.ctx[SwitchDashboardSetZkTask.name]}
            # run switch
            deploy_nanny_task_params = {
                DeployNannyDashboard.ReleaseTask.name: self.id,
                DeployNannyDashboard.NannyDashboardName.name: self.ctx[SwitchDashboardName.name],
                DeployNannyDashboard.NannyDashboardRecipeName.name: self.ctx[SwitchDashboardRecipe.name],
                DeployNannyDashboard.NannyDashboardFilter.name: self.ctx[SwitchDashboardGroups.name],
                DeployNannyDashboard.NannyDashboardItsTask.name: its_values_params,
                DeployNannyDashboard.NannyDashboardSetZkTask.name: set_zk_values_params,
                DeployNannyDashboard.SandboxReleaseType.name: ReleaseStatus.STABLE,
                DeployNannyDashboard.NannyWaitDeployParameter.name: True,
                DeployNannyDashboard.VaultName.name: 'nanny-oauth-token',
                DeployNannyDashboard.VaultOwner.name: 'MEDIA_DEPLOY',
            }
            self.ctx['switch_task_id'] = SandboxTask.create_subtask(
                self,
                task_type=DeployNannyDashboard.DeployNannyDashboard.type,
                description='Switch shardmap: {} task_id: {} task_type: {}'.format(
                    shardmap_filename, self.id, self.type),
                input_parameters=deploy_nanny_task_params,
                inherit_notifications=True
            ).id
            self.set_info("Switching db: {}".format(self.ctx['switch_task_id']))
            if self.ctx.get(MonitoringEnable.name, False):
                self.set_info("Run monitoring task on switching stage")
                self._run_monitoring_task(self.ctx['switch_task_id'], self.ctx[MonitoringSwitchTime.name])

        # Ticket integration logic
        # ========================
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        SandboxTask.on_release(self, additional_parameters)

    def initCtx(self):
        MediaStoreShardmap.initCtx(self)

        # set default values in input params
        self.ctx[ShardmapType.name] = self.SHARDMAP_RESOURCE

    def _generate_shard_name(self, shard, state):
        return self.shardmap_entry('img_commercial', shard, state, 'ImgCommercialTier0')

    def generate_all_shardmap(self, state, meta_state, shardmap_file):
        count = int(self.ctx[PrimaryShardsCount.name])
        for shard in xrange(count):
            shardmap_file.write(self._generate_shard_name(shard, state))


__Task__ = ImagesCommercialStoreShardmap
