# -*- coding: utf-8 -*-

import requests
import copy
import logging

from sandbox import sdk2
from sandbox.projects.MediaLib.media_zk import MediaZkClient

logger = logging.getLogger('ImagesFreezeDeployment')


class HTTPNotFound(Exception):
    pass


class L7heavy(object):
    API_HOST = 'http://ext.its.yandex-team.ru/v2/l7/heavy/'
    REQUEST_HEADERS = {
        'Content-Type': 'application/json; charset=utf-8',
        'Accept': 'application/json',
    }

    def _get_request(self, url, headers=None):
        if headers is None:
            r_headers = dict()
        else:
            r_headers = copy.deepcopy(headers)
        r_headers.update(self.REQUEST_HEADERS)
        r = requests.get("%s" % url, headers=r_headers)
        try:
            r.raise_for_status()
        except requests.exceptions.HTTPError as e:
            if r.status_code == 404:
                raise HTTPNotFound(e)
            else:
                raise Exception(e)
        return r

    def get_weights(self, service_name):
        url = "{}{}/weights/values/".format(self.API_HOST, service_name)
        return self._get_request(url).json()

    def __init__(self, token):
        self.REQUEST_HEADERS.update({
            'Authorization': 'OAuth %s' % token,
        })

class ImagesFreezeDeployment(sdk2.Task):
    """
        Получаем текущие веса по заданным админкам в l7heavy и выставляем указанный флаг в ZK
        при наличии любой закрытой локации. В случае отстутствия таких, удаляем флаг при его
        наличии.
    """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 60 * 5

        l7_heavy_ids = sdk2.parameters.Dict('Список админок l7heavy для мониторинга')
        zk_freeze_flag = sdk2.parameters.String('Флаг в ZK')

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    @staticmethod
    def judge_weights(section_weights):
        """
        :param section_weights: json-объект с весами из nanny
        :return: bool, true — выставить ZK-флаг
        """
        for location, weight in section_weights['locations'].iteritems():
            current_weight = weight['weight']
            default_weight = weight['default_weight']
            if current_weight == 0 and default_weight != 0:
                logger.info('location {} is closed'.format(location))
                return True
        return False

    @staticmethod
    def set_znode(znode, content=None):
        logger.info('setting zk node {} with content {}'.format(znode, content))
        with MediaZkClient() as zk:
            if not zk.exists(znode):
                zk.create(znode, makepath=True)
            zk.set(znode, content)

    @staticmethod
    def del_znode(znode):
        logger.info('removing zk node {}'.format(znode))
        with MediaZkClient() as zk:
            if zk.exists(znode):
                zk.delete(znode)

    def on_execute(self):
        nanny_token = sdk2.Vault.data(self.owner, 'nanny-oauth-token')
        l7heavy = L7heavy(token=nanny_token)
        for service, section_filter in self.Parameters.l7_heavy_ids.iteritems():
            logger.info('get weights for {}'.format(service))
            weights = l7heavy.get_weights(service)
            for section_name, section_object in weights['sections'].iteritems():
                if len(section_filter) > 0 and section_name not in section_filter.split(','):
                    logger.info('filter out section {} in service {}'.format(section_name, service))
                    continue
                logger.info('judge weights for section {} in service {}'.format(section_name, service))
                if self.judge_weights(section_object):
                    # set zk flag
                    self.set_znode(self.Parameters.zk_freeze_flag)
                    return
        self.del_znode(self.Parameters.zk_freeze_flag)
