# -*- coding: utf-8 -*-

import logging

from sandbox.common.types.task import ReleaseStatus, Status

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter
from sandbox.sandboxsdk.parameters import ListRepeater
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sdk2 import yav

from sandbox.projects.common import utils
from sandbox.projects import DeployNannyDashboard, MediaTaskMonitoring
from sandbox.projects.common.nanny import nanny
from sandbox.projects.MediaLib.MediaStoreShardmap import MediaStoreShardmap,\
    ShardmapType, IndexState, UserUploadTaskId

logger = logging.getLogger()


class PrimaryShardsCount(SandboxStringParameter):
    name = 'primary_shards_count'
    description = 'Shards count'
    required = True
    default_value = '2000'


class PrepareDashboardName(SandboxStringParameter):
    name = 'prepare_dashboard_name'
    description = 'Prepare Dashboard name'
    required = True
    default_value = 'images_runtime'
    group = 'prepare'


class PrepareDashboardRecipe(SandboxStringParameter):
    name = 'prepare_dashboard_recipe'
    description = 'Prepare Dashboard recipe'
    required = True
    default_value = 'db_prepare_large_thumb'
    group = 'prepare'


class PrepareDashboardGroups(SandboxStringParameter):
    name = 'prepare_dashboard_groups'
    description = 'Prepare Dashboard groups (comma separated)'
    required = False
    default_value = 'large_thumb_base_nidx'
    group = 'prepare'


class PrepareDashboardItsTask(ListRepeater, SandboxStringParameter):
    name = 'prepare_dashboard_its_task'
    description = 'ITS ручки для обновления версии базы'
    required = False
    default_value = [
        'thumbnails/imgs_improxy/priemka/balancer_improxy_imglth_version/',
    ]
    group = 'prepare'


class PrepareDashboardSetZkTask(ListRepeater, SandboxStringParameter):
    name = 'prepare_dashboard_set_zk_task'
    description = 'ZK ноды для обновления версии базы'
    required = False
    default_value = [
        '/media-services/images/flags/newdb_imglth/on_production',
    ]
    group = 'prepare'


class SwitchDashboardName(SandboxStringParameter):
    name = 'switch_dashboard_name'
    description = 'Switch Dashboard name'
    required = True
    default_value = 'images_runtime'
    group = 'switch'


class SwitchDashboardRecipe(SandboxStringParameter):
    name = 'switch_dashboard_recipe'
    description = 'Switch Dashboard recipe'
    required = True
    default_value = 'db_switch_large_thumb'
    group = 'switch'


class SwitchDashboardGroups(SandboxStringParameter):
    name = 'switch_dashboard_groups'
    description = 'Switch Dashboard groups (comma separated)'
    required = False
    default_value = 'large_thumb_base_prod'
    group = 'switch'


class SwitchDashboardItsTask(ListRepeater, SandboxStringParameter):
    name = 'switch_dashboard_its_task'
    description = 'ITS ручки для обновления версии базы'
    required = False
    default_value = [
        'thumbnails/imgs_improxy/sas/balancer_improxy_imglth_version/',
        'thumbnails/imgs_improxy/msk/balancer_improxy_imglth_version/',
        'thumbnails/imgs_improxy/man/balancer_improxy_imglth_version/',
        'thumbnails/imgs_improxy/vla/balancer_improxy_imglth_version/'
    ]
    group = 'switch'


class SwitchDashboardSetZkTask(ListRepeater, SandboxStringParameter):
    name = 'switch_dashboard_set_zk_task'
    description = 'ZK ноды для обновления версии базы'
    required = False
    default_value = [
        '/media-services/images/flags/production_imglth/on_production',
    ]
    group = 'switch'


class MonitoringEnable(SandboxBoolParameter):
    name = 'monitoring_enable'
    description = 'Enable monitoring'
    default_value = True
    group = 'monitoring'


class MonitoringTelegramChatId(MediaTaskMonitoring.TelegramChatId):
    group = 'monitoring'
    default_value = '-1001088652476'


class MonitoringEmail(MediaTaskMonitoring.Email):
    group = 'monitoring'
    default_value = 'images-newdb@yandex-team.ru'


class MonitoringPrepareTime(MediaTaskMonitoring.MonitoringTime):
    name = 'monitoring_prepare_time'
    group = 'monitoring'
    description = 'Time to alert on preparing stage (seconds)'
    default_value = 12 * 60 * 60  # 12 hours


class MonitoringSwitchTime(MediaTaskMonitoring.MonitoringTime):
    name = 'monitoring_switch_time'
    group = 'monitoring'
    description = 'Time to alert on switching stage (seconds)'
    default_value = 3 * 60 * 60  # 3 hours


class ImagesLargeThumbStoreShardmap(MediaStoreShardmap):
    """Переключение базы тумбов Яндекс.Картинок"""

    type = 'IMAGES_LARGE_THUMB_STORE_SHARDMAP'
    input_parameters = (ShardmapType,
                        PrimaryShardsCount,
                        IndexState,
                        UserUploadTaskId,
                        PrepareDashboardName,
                        PrepareDashboardRecipe,
                        PrepareDashboardGroups,
                        PrepareDashboardItsTask,
                        PrepareDashboardSetZkTask,
                        SwitchDashboardName,
                        SwitchDashboardRecipe,
                        SwitchDashboardGroups,
                        SwitchDashboardItsTask,
                        SwitchDashboardSetZkTask,
                        MonitoringEnable,
                        MonitoringTelegramChatId,
                        MonitoringEmail,
                        MonitoringPrepareTime,
                        MonitoringSwitchTime)


    # task to release for shard
    SHARDMAP_RESOURCE = 'IMAGES_LARGE_THUMB_SHARDMAP'
    SHARD_PREFIX = 'imglth'

    # Send shard size to solomon
    SOLOMON_PROJECT = "images"
    SOLOMON_SERVICE = "bigthumbs"

    YAV_SECRET = "sec-01ehs4x7aqqbd15w2keczwknnq"

    def _run_monitoring_task(self, monitoring_id, monitoring_time):
        task_params = {
            MediaTaskMonitoring.MonitoringTaskId.name: monitoring_id,
            MediaTaskMonitoring.MonitoringTime.name: monitoring_time,
            MediaTaskMonitoring.TelegramChatId.name: self.ctx[MonitoringTelegramChatId.name],
            MediaTaskMonitoring.Email.name: self.ctx[MonitoringEmail.name],
        }
        return SandboxTask.create_subtask(
            self,
            task_type=MediaTaskMonitoring.MediaTaskMonitoring.type,
            description="Monitoring task {}".format(monitoring_id),
            input_parameters=task_params,
            inherit_notifications=True
        ).id

    def _get_nanny_taskgroup_status(self, upload_task_id):
        # check upload task status
        upload_task = channel.sandbox.get_task(upload_task_id)
        if upload_task.new_status not in (Status.SUCCESS,):
            raise SandboxTaskFailureError(
                'Subtask is not OK: {id}'.format(id=upload_task_id)
            )
        upload_taskgroup = upload_task.ctx.get("deploy_taskgroup")
        vault_owner = utils.get_or_default(upload_task.ctx, DeployNannyDashboard.VaultOwner)
        vault_key = utils.get_or_default(upload_task.ctx, DeployNannyDashboard.VaultName)
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data(vault_owner, vault_key),
        )
        return upload_taskgroup, nanny_client.get_taskgroup_status(upload_taskgroup)['status']

    def on_release(self, additional_parameters):
        shardmap_filename = self._generate_shardmap_filename()

        # NewDB Logic (PRESTABLE release)
        # ===============================
        its_values_params = dict()
        set_zk_values_params = dict()
        if additional_parameters['release_status'] == ReleaseStatus.PRESTABLE:
            if self.ctx.get('upload_task_id') is not None:
                raise Exception('Already released to PRESTABLE and created NewDB upload.')

            if self.ctx.get(PrepareDashboardItsTask.name):
                its_values_params = {key: self.ctx[IndexState.name] for key in self.ctx[PrepareDashboardItsTask.name]}
            if self.ctx.get(PrepareDashboardSetZkTask.name):
                set_zk_values_params = {key: shardmap_filename for key in self.ctx[PrepareDashboardSetZkTask.name]}
            # run upload
            upload_task_params = {
                DeployNannyDashboard.ReleaseTask.name: self.id,
                DeployNannyDashboard.NannyDashboardName.name: self.ctx[PrepareDashboardName.name],
                DeployNannyDashboard.NannyDashboardRecipeName.name: self.ctx[PrepareDashboardRecipe.name],
                DeployNannyDashboard.NannyDashboardFilter.name: self.ctx[PrepareDashboardGroups.name],
                DeployNannyDashboard.NannyDashboardItsTask.name: its_values_params,
                DeployNannyDashboard.NannyDashboardSetZkTask.name: set_zk_values_params,
                DeployNannyDashboard.SandboxReleaseType.name: ReleaseStatus.PRESTABLE,
                DeployNannyDashboard.NannyWaitDeployParameter.name: True,
                DeployNannyDashboard.VaultName.name: 'nanny-oauth-token',
                DeployNannyDashboard.VaultOwner.name: 'MEDIA_DEPLOY',
            }
            self.ctx['upload_task_id'] = SandboxTask.create_subtask(
                self,
                task_type=DeployNannyDashboard.DeployNannyDashboard.type,
                description='Upload shardmap: {} task_id: {} task_type: {}'.format(shardmap_filename, self.id, self.type),
                input_parameters=upload_task_params,
                inherit_notifications=True
            ).id
            self.set_info("Start uploading new db: {}".format(self.ctx['upload_task_id']))
            if self.ctx.get(MonitoringEnable.name, False):
                self.set_info("Run monitoring task on preparing stage")
                self._run_monitoring_task(self.ctx['upload_task_id'], self.ctx[MonitoringPrepareTime.name])

        # Switch DB Logic (STABLE release)
        # ================================
        elif additional_parameters['release_status'] == ReleaseStatus.STABLE:
            if self.ctx.get('switch_task_id') is not None:
                raise SandboxTaskFailureError("Task's already been in STABLE")
            # get upload task id
            if self.ctx.get(UserUploadTaskId.name):
                upload_task_id = self.ctx.get(UserUploadTaskId.name)
            elif self.ctx.get('upload_task_id') is not None:
                upload_task_id = self.ctx['upload_task_id']
            else:
                raise SandboxTaskFailureError("You should deploy (upload) database first!")

            # check upload task status
            upload_task_group, upload_status = self._get_nanny_taskgroup_status(upload_task_id)
            if upload_status not in DeployNannyDashboard.ALEMATE_SUCCESS_GROUP:
                taskgroup_url = "https://nanny.yandex-team.ru/ui/#/alemate/taskgroups/{}/table/".format(
                    upload_task_group)
                raise SandboxTaskFailureError('Upload task group has unappropriate status: {}. Link: {}'.format(
                    upload_status, taskgroup_url
                ))

            # send stats
            try:
                secret = yav.Secret(self.YAV_SECRET)
                solomon_token = secret.data()["solomon_token"]
                self.send_stats(self.SHARD_PREFIX, '0000', self.ctx[IndexState.name], self.SOLOMON_PROJECT,
                                self.SOLOMON_SERVICE, solomon_token)
            except Exception as e:
                logger.exception(e)
                self.set_info("<strong>Can't post stats.</strong>", do_escape=False)

            if self.ctx.get(SwitchDashboardItsTask.name):
                its_values_params = {key: self.ctx[IndexState.name] for key in self.ctx[SwitchDashboardItsTask.name]}
            if self.ctx.get(SwitchDashboardSetZkTask.name):
                set_zk_values_params = {key: shardmap_filename for key in self.ctx[SwitchDashboardSetZkTask.name]}
            # run switch
            deploy_nanny_task_params = {
                DeployNannyDashboard.ReleaseTask.name: self.id,
                DeployNannyDashboard.NannyDashboardName.name: self.ctx[SwitchDashboardName.name],
                DeployNannyDashboard.NannyDashboardRecipeName.name: self.ctx[SwitchDashboardRecipe.name],
                DeployNannyDashboard.NannyDashboardFilter.name: self.ctx[SwitchDashboardGroups.name],
                DeployNannyDashboard.NannyDashboardItsTask.name: its_values_params,
                DeployNannyDashboard.NannyDashboardSetZkTask.name: set_zk_values_params,
                DeployNannyDashboard.SandboxReleaseType.name: ReleaseStatus.STABLE,
                DeployNannyDashboard.NannyWaitDeployParameter.name: True,
                DeployNannyDashboard.VaultName.name: 'nanny-oauth-token',
                DeployNannyDashboard.VaultOwner.name: 'MEDIA_DEPLOY',
            }
            self.ctx['switch_task_id'] = SandboxTask.create_subtask(
                self,
                task_type=DeployNannyDashboard.DeployNannyDashboard.type,
                description='Switch shardmap: {} task_id: {} task_type: {}'.format(
                    shardmap_filename, self.id, self.type),
                input_parameters=deploy_nanny_task_params,
                inherit_notifications=True
            ).id
            self.set_info("Switching db: {}".format(self.ctx['switch_task_id']))
            if self.ctx.get(MonitoringEnable.name, False):
                self.set_info("Run monitoring task on switching stage")
                self._run_monitoring_task(self.ctx['switch_task_id'], self.ctx[MonitoringSwitchTime.name])

        # Ticket integration logic
        # ========================
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        SandboxTask.on_release(self, additional_parameters)

    def initCtx(self):
        MediaStoreShardmap.initCtx(self)

        # set default values in input params
        self.ctx[ShardmapType.name] = self.SHARDMAP_RESOURCE

    def shardmap_entry(self, prefix, shard, state, tier):
        template = "{prefix}-{shard:04d}-00000000-000000 {prefix}-{shard:04d}-{state} {tier}\n"
        return template.format(prefix=prefix, shard=shard,
                               state=state, tier=tier)

    def generate_all_shardmap(self, state, meta_state, shardmap_file):
        count = int(self.ctx[PrimaryShardsCount.name])
        for shard in xrange(count):
            shardmap_file.write(self.shardmap_entry(self.SHARD_PREFIX, shard, state, 'ImtubLargeTier0'))


__Task__ = ImagesLargeThumbStoreShardmap
