import datetime

import sandbox.common.types.task as task_types

from sandbox import sdk2
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.cajuper.common import switch, index_state_types

from sandbox.projects import DeployNannyDashboard as deploy_nanny_dashboard
from sandbox.projects import MediaTaskMonitoring as media_task_monitoring


PROJECT = 'IMAGES/RIM'
_VAULT_OWNER = 'MEDIA_DEPLOY'


class ImagesRim3kStoreShardmap(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        db_timestamp = sdk2.parameters.Integer('DB timestamp', required=True)

        with sdk2.parameters.Group("Switch") as switch_block:
            SwitchDashboardName = sdk2.parameters.String('Switch Dashboard name', default='images_rim3k', required=True)
            SwitchDashboardRecipe = sdk2.parameters.String('Switch Dashboard recipe', default='db_switch_rim3k', required=True)
            SwitchDashboardGroups = sdk2.parameters.String('Switch Dashboard groups (comma separated)', default='rim_base_prod')
            SwitchDashboardItsTask = sdk2.parameters.List(
                'ITS knobs to update database version',
                default=[
                    'thumbnails/rim/sas/balancer_rimdb_version/',
                    'thumbnails/rim/man/balancer_rimdb_version/',
                    'thumbnails/rim/vla/balancer_rimdb_version/',
                ],
            )
            SemaphoreName = sdk2.parameters.String('Semephore name', default='IMAGES_RIM3K_DEPLOY', required=True)

        with sdk2.parameters.Group("Monitoring") as monitoring_block:
            email = sdk2.parameters.String('Send emails', default='images-newdb@yandex-team.ru')
            telegram_chat_id = sdk2.parameters.String('Chat id for telegram', default='-1001088652476')
            switch_timeout = sdk2.parameters.Integer('Time to alert in seconds', default=6 * 60 * 60)

    def on_enqueue(self):
        self.Context.db_state = index_state_types.timestamp_to_yt_state(self.Parameters.db_timestamp)

    def on_execute(self):
        with self.memoize_stage.run_state_resource_creator(commit_on_entrance=False):
            sub_task = switch.run_state_creator(self, self.Parameters.db_timestamp, PROJECT)

            self.Context.state_creator_id = sub_task.id

            raise sdk2.WaitTask(
                [self.Context.state_creator_id],
                task_types.Status.Group.FINISH | task_types.Status.Group.BREAK,
                wait_all=True
            )

        with self.memoize_stage.get_state_resource(commit_on_entrance=False):
            state_resources = switch.copy_index_state_resources(self, self.Context.state_creator_id, PROJECT)
            state_resources.pop().ready()

    def on_release(self, additional_parameters):
        with self.memoize_stage.nanny_release(commit_on_entrance=False):
            nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)

        if additional_parameters['release_status'] == task_types.ReleaseStatus.STABLE:
            with self.memoize_stage.deployment(commit_on_entrance=False):
                self.Context.deployment_task_id = self.__run_deployment_task().id
                self.set_info(
                    'Created deployment task <a href="/task/{id}">{id}</a>'.format(
                        id=self.Context.deployment_task_id
                    ),
                    do_escape=False
                )

            with self.memoize_stage.monitoring(commit_on_entrance=False):
                self.Context.monitoring_task_id = self.__run_monitoring_task(self.Context.deployment_task_id).id
                self.set_info(
                    'Created monitoring task <a href="/task/{id}">{id}</a>'.format(
                        id=self.Context.monitoring_task_id
                    ),
                    do_escape=False
                )

        with self.memoize_stage.do_release(commit_on_entrance=False):
            sdk2.Task.on_release(self, additional_parameters)

    def __run_deployment_task(self):
        sub_ctx = {
            deploy_nanny_dashboard.ReleaseTask.name: self.id,
            deploy_nanny_dashboard.NannyDashboardName.name: self.Parameters.SwitchDashboardName,
            deploy_nanny_dashboard.NannyDashboardRecipeName.name: self.Parameters.SwitchDashboardRecipe,
            deploy_nanny_dashboard.SandboxReleaseType.name: task_types.ReleaseStatus.STABLE,
            deploy_nanny_dashboard.NannyDashboardItsTask.name: {
                its_values: self.Context.db_state
                for its_values in self.Parameters.SwitchDashboardItsTask
            },
            deploy_nanny_dashboard.WaitDeployment.name: 5 * 60,
            deploy_nanny_dashboard.NannyWaitDeployParameter.name: True,
            deploy_nanny_dashboard.GetServicesFromRecipe.name: True,
            deploy_nanny_dashboard.VaultOwner.name: _VAULT_OWNER,
            deploy_nanny_dashboard.SemaphoreName.name: self.Parameters.SemaphoreName
        }
        sub_task_class = sdk2.Task[deploy_nanny_dashboard.DeployNannyDashboard.type]
        return sub_task_class(self, **sub_ctx).enqueue()

    def __run_monitoring_task(self, monitoring_id):
        sub_ctx = {
            media_task_monitoring.MonitoringTaskId.name: monitoring_id,
            media_task_monitoring.TelegramChatId.name: self.Parameters.telegram_chat_id,
            media_task_monitoring.Email.name: self.Parameters.email,
            media_task_monitoring.MonitoringTime.name: str(self.Parameters.switch_timeout),
            media_task_monitoring.MonitoringSleep.name: 5 * 60,
            media_task_monitoring.VaultOwner.name: _VAULT_OWNER,
        }
        sub_task_class = sdk2.Task[media_task_monitoring.MediaTaskMonitoring.type]
        return sub_task_class(self, **sub_ctx).enqueue()


def yt_state_to_timestamp(yt_state):
    return int(datetime.datetime.strptime(yt_state, '%Y%m%d-%H%M%S').strftime('%s'))
