from sandbox.common.types.task import ReleaseStatus, Status

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects import DeployNannyDashboard as deploy_nanny_dashboard
from sandbox.projects import MediaTaskMonitoring as media_task_monitoring
from sandbox.projects.common.nanny import nanny
from sandbox.projects.MediaLib import MediaStoreShardmap as media_store_shardmap
from sandbox.projects.images.deployment import resources as deployment_resources


_SWITCH_TASK_ID = 'switch_task_id'
_PREPARE_TASK_ID = 'prepare_task_id'

_MONITORING_PARAMS = media_task_monitoring.create_monitoring_params(
    email="images-newdb@yandex-team.ru",
    telegram_chat_id='-1001088652476',
    switch_timeout=6 * 60 * 60,
)
_MONITORING_SWITCH_TASK_ID = 'monitoring_switch_task_id'
_MONITORING_PREPARE_TASK_ID = 'monitoring_prepare_task_id'

_STABLE_RELEASE = ReleaseStatus.STABLE
_PRESTABLE_RELEASE = ReleaseStatus.PRESTABLE


class PrimaryShardsCount(parameters.SandboxStringParameter):
    name = 'primary_shards_count'
    description = 'Shards count'
    required = True
    default_value = '500'


class SwitchDashboardName(parameters.SandboxStringParameter):
    name = 'switch_dashboard_name'
    description = 'Switch Dashboard name'
    required = True
    default_value = 'images_runtime'
    group = 'switch'


class SwitchDashboardRecipe(parameters.SandboxStringParameter):
    name = 'switch_dashboard_recipe'
    description = 'Switch Dashboard recipe'
    required = True
    default_value = 'db_switch_rim'
    group = 'switch'


class SwitchDashboardGroups(parameters.SandboxStringParameter):
    name = 'switch_dashboard_groups'
    description = 'Switch Dashboard groups (comma separated)'
    required = False
    default_value = 'rim_base_prod'
    group = 'switch'


class SwitchDashboardSetZkTask(parameters.ListRepeater, parameters.SandboxStringParameter):
    name = 'switch_dashboard_set_zk_task'
    description = 'ZK nodes to update database version'
    required = False
    default_value = [
        '/media-services/images/flags/production_rim_msk/on_production',
    ]
    group = 'switch'


class SwitchDashboardItsTask(parameters.ListRepeater, parameters.SandboxStringParameter):
    name = 'deployment_nanny_dashboard_its_task'
    description = 'ITS knobs to update database version'
    required = False
    default_value = [
        'thumbnails/rim/sas/balancer_rimdb_version/',
        'thumbnails/rim/msk/balancer_rimdb_version/',
        'thumbnails/rim/man/balancer_rimdb_version/',
        'thumbnails/rim/vla/balancer_rimdb_version/'
    ]
    group = 'switch'


class NewdbDashboardName(parameters.SandboxStringParameter):
    name = 'newdb_dashboard_name'
    description = 'Switch Dashboard name'
    required = True
    default_value = 'images_runtime'
    group = 'newdb'


class NewdbDashboardRecipe(parameters.SandboxStringParameter):
    name = 'newdb_dashboard_recipe'
    description = 'Newdb Dashboard recipe'
    required = True
    default_value = 'db_prepare_rim'
    group = 'newdb'


class NewdbDashboardGroups(parameters.SandboxStringParameter):
    name = 'newdb_dashboard_groups'
    description = 'Newdb Dashboard groups (comma separated)'
    required = False
    default_value = 'rim_base_nidx'
    group = 'newdb'


class NewdbDashboardSetZkTask(parameters.ListRepeater, parameters.SandboxStringParameter):
    name = 'newdb_dashboard_set_zk_task'
    description = 'ZK nodes to update database version'
    required = False
    default_value = [
        '/media-services/images/flags/acceptance_rim_sas/on_production',
    ]
    group = 'newdb'


class NewdbDashboardItsTask(parameters.ListRepeater, parameters.SandboxStringParameter):
    name = 'newdb_deployment_nanny_dashboard_its_task'
    description = 'ITS knobs to update database version'
    required = False
    default_value = [
        'thumbnails/rim/priemka/balancer_rimdb_version/',
    ]
    group = 'newdb'


class ImagesRimStoreShardmap(nanny.ReleaseToNannyTask, media_store_shardmap.MediaStoreShardmap):
    """Switch Yandex.Images RIM database"""

    type = 'IMAGES_RIM_STORE_SHARDMAP'
    input_parameters = (
                           PrimaryShardsCount,
                           media_store_shardmap.IndexState,
                           media_store_shardmap.UserUploadTaskId,
                           NewdbDashboardRecipe,
                           NewdbDashboardName,
                           NewdbDashboardGroups,
                           NewdbDashboardSetZkTask,
                           NewdbDashboardItsTask,
                           SwitchDashboardRecipe,
                           SwitchDashboardName,
                           SwitchDashboardGroups,
                           SwitchDashboardSetZkTask,
                           SwitchDashboardItsTask
    ) + _MONITORING_PARAMS
    shardmap_resource = deployment_resources.IMAGES_RIM_SHARDMAP

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

        if additional_parameters['release_status'] == _STABLE_RELEASE:
            if self.ctx.get(_PREPARE_TASK_ID):
                newdb_task_id = self.ctx.get(_PREPARE_TASK_ID)
            elif self.ctx.get(media_store_shardmap.UserUploadTaskId.name) is not None:
                newdb_task_id = self.ctx[media_store_shardmap.UserUploadTaskId.name]
            else:
                raise SandboxTaskFailureError("You should deploy (upload) database first!")
            # check upload task status
            newdb_task = channel.sandbox.get_task(newdb_task_id)
            if newdb_task.new_status not in (Status.SUCCESS,):
                raise SandboxTaskFailureError('Subtask is not ok: {id}'.format(id=newdb_task_id))

            if _SWITCH_TASK_ID not in self.ctx:
                self.ctx[_SWITCH_TASK_ID] = self._run_deployment_task(
                    dashboard_name=self.ctx[SwitchDashboardName.name],
                    dashboard_recipe=self.ctx[SwitchDashboardRecipe.name],
                    dashboard_groups=self.ctx[SwitchDashboardGroups.name],
                    release_type=_STABLE_RELEASE,
                    zk_values=self.ctx[SwitchDashboardSetZkTask.name],
                    its_values=self.ctx[SwitchDashboardItsTask.name]
                )
                self.set_info("Switching db: {}".format(self.ctx[_SWITCH_TASK_ID]))
            if _MONITORING_SWITCH_TASK_ID not in self.ctx:
                self.ctx[_MONITORING_SWITCH_TASK_ID] = self._run_monitoring_task(self.ctx[_SWITCH_TASK_ID])
                self.set_info("Monitor switch deployment: {}".format(self.ctx[_MONITORING_SWITCH_TASK_ID]))
        elif additional_parameters['release_status'] == _PRESTABLE_RELEASE:
            if _PREPARE_TASK_ID not in self.ctx:
                self.ctx[_PREPARE_TASK_ID] = self._run_deployment_task(
                    dashboard_name=self.ctx[NewdbDashboardName.name],
                    dashboard_recipe=self.ctx[NewdbDashboardRecipe.name],
                    dashboard_groups=self.ctx[NewdbDashboardGroups.name],
                    release_type=_PRESTABLE_RELEASE,
                    zk_values=self.ctx[NewdbDashboardSetZkTask.name],
                    its_values=self.ctx[NewdbDashboardItsTask.name]
                )
                self.set_info("Preparing db: {}".format(self.ctx[_PREPARE_TASK_ID]))
            if _MONITORING_PREPARE_TASK_ID not in self.ctx:
                self.ctx[_MONITORING_PREPARE_TASK_ID] = self._run_monitoring_task(self.ctx[_PREPARE_TASK_ID])
                self.set_info("Monitor prepare deployment: {}".format(self.ctx[_MONITORING_PREPARE_TASK_ID]))

    def generate_all_shardmap(self, state, meta_state, shardmap_file):
        count = int(self.ctx[PrimaryShardsCount.name])
        for shard in range(count):
            shardmap_file.write(self.shardmap_entry('imgsrim', shard, state, 'ImgRimTier0'))

    def _run_monitoring_task(self, monitoring_id):
        sub_ctx = {param.name: self.ctx[param.name] for param in _MONITORING_PARAMS}
        sub_ctx.update({
            media_task_monitoring.MonitoringTaskId.name:  monitoring_id,
        })
        return self.create_subtask(
            task_type=media_task_monitoring.MediaTaskMonitoring.type,
            description="Monitoring task {}".format(monitoring_id),
            input_parameters=sub_ctx,
            inherit_notifications=True
        ).id

    def _run_deployment_task(self,
                             dashboard_name,
                             dashboard_recipe,
                             dashboard_groups,
                             release_type,
                             zk_values=None,
                             its_values=None):
        shardmap_filename = self._generate_shardmap_filename()

        its_values_params = dict()
        if its_values:
            its_values_params = {
                key: self.ctx[media_store_shardmap.IndexState.name]
                for key in its_values
            }

        set_zk_values_params = dict()
        if zk_values:
            set_zk_values_params = {
                key: shardmap_filename
                for key in zk_values
            }

        deploy_nanny_task_params = {
            deploy_nanny_dashboard.ReleaseTask.name: self.id,
            deploy_nanny_dashboard.NannyDashboardName.name: dashboard_name,
            deploy_nanny_dashboard.NannyDashboardRecipeName.name: dashboard_recipe,
            deploy_nanny_dashboard.NannyDashboardFilter.name: dashboard_groups,
            deploy_nanny_dashboard.NannyDashboardItsTask.name: its_values_params,
            deploy_nanny_dashboard.NannyDashboardSetZkTask.name: set_zk_values_params,
            deploy_nanny_dashboard.SandboxReleaseType.name: release_type,
            deploy_nanny_dashboard.NannyWaitDeployParameter.name: True,
            deploy_nanny_dashboard.VaultName.name: 'nanny-oauth-token',
            deploy_nanny_dashboard.VaultOwner.name: 'MEDIA_DEPLOY',
        }
        return self.create_subtask(
            task_type=deploy_nanny_dashboard.DeployNannyDashboard.type,
            description='Waiting for uploaded shardmap: {} task_id: {}, task_type: {}'.format(
                shardmap_filename,
                self.id,
                self.type),
            input_parameters=deploy_nanny_task_params,
            inherit_notifications=True
        ).id


__Task__ = ImagesRimStoreShardmap
