import sandbox.common.types.task as ctt

from sandbox.sandboxsdk import parameters

from sandbox.projects import DeployNannyDashboard as deploy_nanny_dashboard
from sandbox.projects import MediaTaskMonitoring as media_task_monitoring
from sandbox.projects.common.nanny import nanny
from sandbox.projects.MediaLib import MediaStoreShardmap as media_store_shardmap
from sandbox.projects.images.deployment import resources as deployment_resources


_SWITCH_PARAMS = deploy_nanny_dashboard.create_switching_params(
    dashboard_name='images_rq2',
    dashboard_recipe='db_switch_rq',
    dashboard_filter='base'
)
_SWITCH_TASK_ID = 'switch_task_id'

_MONITORING_PARAMS = media_task_monitoring.create_monitoring_params(
    email="images-newdb@yandex-team.ru",
    telegram_chat_id="-1001088652476",
    switch_timeout=3 * 60 * 60,
)
_MONITORING_TASK_ID = 'monitoring_task_id'

_STABLE_RELEASE = ctt.ReleaseStatus.STABLE


class PrimaryShardsCount(parameters.SandboxIntegerParameter):
    name = 'primary_shards_count'
    description = 'Shards count'
    default_value = 10
    required = True


class ImagesRqStoreShardmap(nanny.ReleaseToNannyTask, media_store_shardmap.MediaStoreShardmap):
    """Switch database in related queries (Yandex.Images)"""

    type = 'IMAGES_RQ_STORE_SHARDMAP'

    input_parameters = (
        media_store_shardmap.IndexState,
        PrimaryShardsCount,
    ) + _SWITCH_PARAMS + _MONITORING_PARAMS

    shardmap_resource = deployment_resources.IMAGES_RQ_SHARDMAP

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

        # Switch DB Logic (STABLE release)
        if additional_parameters['release_status'] == _STABLE_RELEASE:
            if _SWITCH_TASK_ID not in self.ctx:
                self.ctx[_SWITCH_TASK_ID] = self._run_deployment_task()
                self.set_info("Switching db: {}".format(self.ctx[_SWITCH_TASK_ID]))
            if _MONITORING_TASK_ID not in self.ctx:
                self.ctx[_MONITORING_TASK_ID] = self._run_monitoring_task(self.ctx[_SWITCH_TASK_ID])
                self.set_info("Monitor deployment: {}".format(self.ctx[_MONITORING_TASK_ID]))

    def _run_deployment_task(self):
        shardmap_filename = self._generate_shardmap_filename()
        sub_ctx = {param.name: self.ctx[param.name] for param in _SWITCH_PARAMS}
        sub_ctx.update({
            deploy_nanny_dashboard.ReleaseTask.name: self.id,
            deploy_nanny_dashboard.SandboxReleaseType.name: _STABLE_RELEASE,
            deploy_nanny_dashboard.NannyWaitDeployParameter.name: True,
        })
        return self.create_subtask(
            task_type=deploy_nanny_dashboard.DeployNannyDashboard.type,
            description='Deployment of {}, task_id: {}'.format(shardmap_filename, self.id),
            input_parameters=sub_ctx,
            inherit_notifications=True
        ).id

    def _run_monitoring_task(self, monitoring_id):
        sub_ctx = {param.name: self.ctx[param.name] for param in _MONITORING_PARAMS}
        sub_ctx.update({
            media_task_monitoring.MonitoringTaskId.name:  monitoring_id,
        })
        self.create_subtask(
            task_type=media_task_monitoring.MediaTaskMonitoring.type,
            description='Monitoring of {}, task_id: {}'.format(monitoring_id, self.id),
            input_parameters=sub_ctx,
            inherit_notifications=True
        ).id

    def generate_all_shardmap(self, state, meta_state, shardmap_file):
        for shard_num in xrange(self.ctx[PrimaryShardsCount.name]):
            shardmap_file.write(self.shardmap_entry('imgsrq2baseidx', shard_num, state, 'ImgRQ2BaseTier0'))


__Task__ = ImagesRqStoreShardmap
