import logging

from sandbox import sdk2

import sandbox.common.types.task as task_types

from sandbox.common import errors
from sandbox.projects import DeployNannyDashboard as deploy_nanny_dashboard
from sandbox.projects import MediaTaskMonitoring as media_task_monitoring
from sandbox.projects.images.deployment import zookeeper
from sandbox.projects.common.decorators import retries
from sandbox.projects.common.nanny import nanny

from sandbox.projects.MediaLib.media_zk import MediaZkClient

logger = logging.getLogger()

RETRIES = 10
DELAY = 30


class ImagesSwitchBinary(sdk2.Task):
    """Switch binary on production using meta recipe"""

    ZK_SYNC_NODE = '/media-services/images/flags/deployment/branch_deployment'

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        release_task = sdk2.parameters.Task(
            "Task with release resources",
            required=True
        )
        vault_owner = sdk2.parameters.String(
            "Sandbox vault owner",
            default="MEDIA_DEPLOY",
            required=True
        )
        nanny_dashboard_name = sdk2.parameters.String(
            "Nanny dashboard for autodeploy",
            required=True
        )
        nanny_dashboard_recipe = sdk2.parameters.String(
            "Nanny dashboard recipe",
            required=True
        )
        monitoring_chat = sdk2.parameters.String(
            "Telegram chat id",
            default="-1001088652476",
            required=True
        )
        monitoring_email = sdk2.parameters.String(
            "Send emails",
            default="images-releases@yandex-team.ru",
            required=True
        )
        alert_time = sdk2.parameters.Integer(
            "Time to alert in seconds",
            default=12 * 60 * 60,
            required=True
        )
        sleep_time = sdk2.parameters.Integer(
            "Time to sleep between monitoring checks",
            default=5 * 60,
        )
        zookeeper_path = sdk2.parameters.String(
            "Zookeper node to lock",
            required=True
        )
        zookeeper_force = sdk2.parameters.Bool(
            "Remove node before start (experts only)",
            default=False
        )
        semaphore_name = sdk2.parameters.String(
            "Semaphore name",
            required=False
        )
        nanny_ticket = sdk2.parameters.String(
            "Ticket id",
            required=False
        )
        zk_sync_node = sdk2.parameters.String(
            "ZK sync node",
            required=False
        )

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def __close_nanny_ticket(self, ticket_id):
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=sdk2.Vault.data('MEDIA_DEPLOY', 'nanny-oauth-token')
        )
        nanny_client.update_ticket_status(ticket_id, 'DEPLOY_SUCCESS', 'Branch switch succeeded!')

    def __update_nanny_ticket(self):
        with MediaZkClient() as zk:
            zk_sync_node = getattr(self.Parameters, 'zk_sync_node', self.ZK_SYNC_NODE)
            content = zk.get(zk_sync_node)[0]
            if int(content) == 1:
                self.__close_nanny_ticket(self.Parameters.nanny_ticket)
                self.set_info("Nanny ticket closed: {}".format(self.Parameters.nanny_ticket))
                zk.delete(zk_sync_node)
            else:
                zk_value = str(int(content) - 1)
                zk.set(zk_sync_node, zk_value)

    def on_finish(self, prev_status, status):
        path = self.Parameters.zookeeper_path
        if path:
            zookeeper.zk_delete(path)

        # close ticket
        if self.Parameters.nanny_ticket:
            try:
                self.__update_nanny_ticket()
            except Exception as e:
                logger.exception("Failed to close nanny ticket: {}".format(e))
                self.set_info("<strong>Can't close deploy ticket {}.</strong>".format(self.Parameters.nanny_ticket),
                              do_escape=False)

    def on_execute(self):
        # Code below is an emulation of 'fail_on_any_error' with a more simple and correct behaviour
        try:
            self.__on_execute()
        except (errors.TaskFailure, sdk2.WaitTask):
            raise
        except Exception as e:
            raise errors.TaskFailure("Unexpected exception: {}".format(e))

    def __on_execute(self):
        with self.memoize_stage.lock(commit_on_entrance=False):
            path = self.Parameters.zookeeper_path
            if path:
                if self.Parameters.zookeeper_force:
                    zookeeper.zk_delete(path)
                zookeeper.zk_create_exclusive(path, value=str(self.id))

        with self.memoize_stage.deployment(commit_on_entrance=False):
            self.Context.deployment_task_id = self.__run_deployment_task().id

        with self.memoize_stage.monitoring(commit_on_entrance=False):
            self.Context.monitoring_task_id = self.__run_monitoring_task(self.Context.deployment_task_id).id

        sub_task_list = [
            sdk2.Task[task_id] for task_id in (self.Context.deployment_task_id, self.Context.monitoring_task_id)
        ]
        sub_task_statuses = task_types.Status.Group.FINISH + task_types.Status.Group.BREAK

        for sub_task in sub_task_list:
            if sub_task.status not in sub_task_statuses:
                raise sdk2.WaitTask(sub_task_list, sub_task_statuses)

        for sub_task in sub_task_list:
            if sub_task.status != task_types.Status.SUCCESS:
                raise errors.TaskFailure("Subtask failed")

    def __run_deployment_task(self):
        sub_ctx = {
            deploy_nanny_dashboard.ReleaseTask.name: int(self.Parameters.release_task),
            deploy_nanny_dashboard.NannyDashboardName.name: self.Parameters.nanny_dashboard_name,
            deploy_nanny_dashboard.NannyDashboardRecipeName.name: self.Parameters.nanny_dashboard_recipe,
            deploy_nanny_dashboard.SandboxReleaseType.name: task_types.ReleaseStatus.STABLE,
            deploy_nanny_dashboard.WaitDeployment.name: self.Parameters.sleep_time,
            deploy_nanny_dashboard.NannyWaitDeployParameter.name: True,
            deploy_nanny_dashboard.GetServicesFromRecipe.name: True,
            deploy_nanny_dashboard.VaultOwner.name: self.Parameters.vault_owner,
            deploy_nanny_dashboard.SemaphoreName.name: getattr(self.Parameters, 'semaphore_name', None)
        }
        sub_task_class = sdk2.Task[deploy_nanny_dashboard.DeployNannyDashboard.type]
        return sub_task_class(self, **sub_ctx).enqueue()

    def __run_monitoring_task(self, monitoring_id):
        sub_ctx = {
            media_task_monitoring.MonitoringTaskId.name: monitoring_id,
            media_task_monitoring.TelegramChatId.name: self.Parameters.monitoring_chat,
            media_task_monitoring.Email.name: self.Parameters.monitoring_email,
            media_task_monitoring.MonitoringTime.name: str(self.Parameters.alert_time),
            media_task_monitoring.MonitoringSleep.name: self.Parameters.sleep_time,
            media_task_monitoring.VaultOwner.name: self.Parameters.vault_owner,
        }
        sub_task_class = sdk2.Task[media_task_monitoring.MediaTaskMonitoring.type]
        return sub_task_class(self, **sub_ctx).enqueue()
