# -*- coding: utf-8 -*-

import logging
import requests
import json

import sandbox.projects.MediaLib.SwitchMediaDatabase as switch_database
from sandbox.projects.common.decorators import retries


logger = logging.getLogger()

SERVICES_FROM_RECIPE = True

SWITCH_WITH_GARBAGE_RECIPE = 'switch_images_database_with_tier1'

ZK_FLAGS = {
    'newdb': '/media-services/images/flags/newdb/newdb_uploaded',
    'on_production': '/media-services/images/flags/newdb/on_production',
    'garbage_on_production': '/media-services/images/flags/newdb/garbage_on_production',
    'thumb': '/media-services/images/flags/production_msk_imgsthwide/on_production',
}

STARTREK_API_URL = "https://st-api.yandex-team.ru/v2/"
STARTREK_UI_URL = "https://st.yandex-team.ru/"


class SwitchImagesDatabaseNG(switch_database.SwitchMediaDatabase):
    """Switching the Main Images Database"""

    type = 'SWITCH_IMAGES_DATABASE_NG'
    switch_type = 'images'

    # YDL
    YDL_TOKEN_NAME = 'ydl_images_token'

    # Monitoring settings
    monitoring_sleep = 10 * 60  # 10 min
    monitoring_time = 7 * 60 * 60  # 7 hours
    monitoring_telegram_chat_id = '-1001088652476'
    monitoring_email_to = 'images-newdb@yandex-team.ru'
    monitoring_vault_name = 'telegram_token'
    monitoring_vault_owner = 'IMAGES-BASE-DEPLOY'
    monitoring_success_alert = False

    # Startrek settings
    startrek_vault_name = 'images_st_token'
    startrek_vault_owner = 'MEDIA_DEPLOY'

    def initCtx(self):
        switch_database.SwitchMediaDatabase.initCtx(self)

        # override inputs
        self.ctx[switch_database.ServicesFromRecipe.name] = SERVICES_FROM_RECIPE
        if self.ctx[switch_database.SwitchDashboardRecipe.name] == SWITCH_WITH_GARBAGE_RECIPE:
            self.ctx[switch_database.IsGarbageDashboardRecipe.name] = True
        self.ctx[switch_database.ZkFlags.name] = ZK_FLAGS

    def on_prepare(self):
        if self.ctx[switch_database.SwitchDashboardRecipe.name] == SWITCH_WITH_GARBAGE_RECIPE:
            self.ctx[switch_database.IsGarbageDashboardRecipe.name] = True

    def _is_thumbs_ready(self):
        """Thumbs readiness is checked in sanity_checks() in this class"""
        return True

    @staticmethod
    def __startrek_headers(startrek_token):
        return {
            "Content-Type": "application/json; charset=utf-8",
            "Authorization": "OAuth {}".format(startrek_token)
        }

    @classmethod
    @retries(max_tries=switch_database.RETRIES, delay=switch_database.DELAY)
    def __index_tickets(cls, index_state, startrek_token):
        """
        Fetch list of Startrek tickets connected with index_state

        Relies on acceptance Startrek ticket's title.

        :return: {ticket_id: ticket_status, ...}
        """

        query = u"Tags: приёмка-индекса" \
            u" AND Queue: IMAGES" \
            u" AND Summary: \"Приёмка индекса {}\"".format(index_state)
        data = {
            "query": query
        }
        data = json.dumps(data)
        logger.debug("Startrek request data: %s", data)

        r = requests.post(
            STARTREK_API_URL + "issues/_search",
            headers=cls.__startrek_headers(startrek_token),
            data=data,
            verify=False
        )

        logger.debug("Startrek response data: %s", r.content)
        r.raise_for_status()

        return {ticket["key"]: ticket["status"]["key"] for ticket in r.json()}

    @classmethod
    @retries(max_tries=switch_database.RETRIES, delay=switch_database.DELAY)
    def __set_ticket_status(cls, ticket, status, startrek_token):
        """Startrek ticket status transition"""

        r = requests.post(
            STARTREK_API_URL + "issues/" + ticket + "/transitions/" + status + "/_execute",
            headers=cls.__startrek_headers(startrek_token),
            verify=False
        )

        logger.debug("Startrek response data: %s", r.content)
        r.raise_for_status()

    def __check_index_acceptance(self, index_state, startrek_token):
        """
        Check if index is accepted

        Acceptance includes thumbs prod switch checks and more.
        Relies on Startrek ticket data.
        Raises Exception if problem is detected.
        """

        tickets = self.__index_tickets(index_state, startrek_token)
        index_is_accepted = any([status in ("readyToDeploy", "deployed") for status in tickets.itervalues()])

        if not index_is_accepted:
            def ticket_info(ticket, status):
                return "<a href=\"{0}{1}\">{1}</a> <i>({2})</i>".format(STARTREK_UI_URL, ticket, status)

            helper_message = "No accepted Startrek tickets found. Related tickets:\n"
            helper_message += "\n".join([ticket_info(ticket, status) for ticket, status in tickets.iteritems()])

            self.set_info(helper_message, do_escape=False)

            raise Exception("Index is NOT accepted! Related tickets: {}".format(", ".join(tickets.iterkeys())))
        else:
            self.set_info("Index is accepted")

    def sanity_checks(self):
        """Images-specific index checks"""

        self.set_info("Starting images-specific checks")

        index_state = self._get_state_from_shardmap(self.ctx[switch_database.ShardmapName.name])
        startrek_token = self.get_vault_data(self.startrek_vault_owner, self.startrek_vault_name)

        self.__check_index_acceptance(index_state, startrek_token)

        self.set_info("Images-specific checks done")

    def deploy_success_callback(self):
        """Images-specific deploy_success actions"""
        self.set_info("Starting images-specific deploy_success procedures")

        index_state = self._get_state_from_shardmap(self.ctx[switch_database.ShardmapName.name])
        startrek_token = self.get_vault_data(self.startrek_vault_owner, self.startrek_vault_name)

        for ticket, status in self.__index_tickets(index_state, startrek_token).iteritems():
            if status == "readyToDeploy":
                # Continue with remaining tickets even if retries fail
                try:
                    self.__set_ticket_status(ticket, "deployed", startrek_token)
                except Exception as e:
                    logger.exception("Failed to switch Startrek ticket {} status to \"deployed\": {}".format(ticket, e))
                    self.set_info("Problem marking {} as deployed".format(ticket))

        self.set_info("Images-specific deploy_success procedures finished")


__Task__ = SwitchImagesDatabaseNG
