import kazoo.client
import kazoo.exceptions
import logging
import random
import time

from sandbox.projects.MediaLib.media_zk import MediaZkClient

def zk_create_exclusive(path, value="", timeout=2*60*60):
    """
        Try to create non ephemeral node exclisively

        Repeat operation until success or timeout. If force flag specified we
        will try to remove existing node
    """

    logging.info("Creating node {} with value '{}'".format(path, value))
    with MediaZkClient() as zk:
        waitfor = time.time() + timeout
        while time.time() < waitfor:
            try:
                zk.create(path, value=value, makepath=True)
                logging.info("Node {} with value '{}' created".format(path, value))
                break
            except kazoo.exceptions.NodeExistsError:
                # Information about current value may be useful for futher debugging
                try:
                    existing_value, existing_stat = zk.get(path)
                    logging.info("Node {} exists and contains value '{}'".format(path, existing_value))
                except kazoo.exceptions.NoNodeError:
                    pass

                logging.info("Node {} exists, try again".format(path))
                time.sleep(random.randint(2, 10))
            except kazoo.exceptions.ZookeeperError as e:
                logging.error("Zookeeper error: {}".format(e))
                time.sleep(random.randint(2, 10))
        else:
            raise Exception("Failed to create node {} in {} seconds".format(path, timeout))


def zk_delete(path, timeout=60):
    """
        Remove node

        Repeat operation until success or timeout
    """

    logging.info("Removing node {}".format(path))
    with MediaZkClient() as zk:
        waitfor = time.time() + timeout
        while time.time() < waitfor:
            try:
                zk.delete(path)
                logging.info("Node {} removed".format(path))
                break
            except kazoo.exceptions.NoNodeError:
                logging.info("Nothing to remove at path {}".format(path))
                break
            except kazoo.exceptions.ZookeeperError as e:
                logging.error("Zookeeper error: {}".format(e))
                time.sleep(random.randint(1, 5))
        else:
            raise Exception("Failed to remove node {} in {} seconds".format(path, timeout))
