# -*- coding: utf-8 -*-

import json
import logging
import requests
from datetime import datetime

from sandbox import sdk2
from sandbox.common.errors import TaskFailure


DEFAULT_ABC_SERVICE_ID = 90


class API(object):

    def __init__(self, token=None):
        self._session = requests.Session()
        self._headers = {}
        if token is not None:
            self._headers["Authorization"] = "OAuth {}".format(token)

    def _request(self, url, method="GET", params=None, data=None):
        request = requests.Request(method, url,
            headers=self._headers,
            params=params,
            data=data,
        )
        try:
            response = self._session.send(request.prepare())
            response.raise_for_status()
        except Exception as e:
            raise TaskFailure("{} '{}' request failed:\n{}".format(method, url, str(e)))
        try:
            return response.json()
        except Exception as e:
            raise TaskFailure("API result parse failed")



class ABC(API):

    API_BASE_URL = 'https://abc-back.yandex-team.ru/api/v4'

    def __init__(self, token):
        super(ABC, self).__init__(token)
        self._headers["Content-Type"] = "application/json";

    def get_duty_logins(self, service_id, date):
        params = {
            'service': service_id,
            'date_from': date,
            'date_to': date
        }

        response = self._request(self.API_BASE_URL + "/duty/shifts/", params=params)

        on_duty = {}
        for duty in response['results']:
            if duty['schedule']['name'] == 'DevOps Service Duty':
                on_duty['first_line'] = duty['person']['login']
            if duty['schedule']['name'] == 'DevOps Service Duty Second Line':
                on_duty['second_line'] = duty['person']['login']

        if 'second_line' not in on_duty or 'first_line' not in on_duty:
            raise TaskFailure("Error on parsing abc response:\n{}".format(response))

        return on_duty


class Staff(API):

    API_BASE_URL = 'https://staff-api.yandex-team.ru/v3'

    def get_person_info(self, login, fields=None):
        params = {
            "login": login,
        }
        if fields is not None:
            params["_fields"] = ",".join(fields)
        result = self._request(self.API_BASE_URL + "/persons", params=params)
        return next(iter(result["result"]), None)


class Telegraph(API):

    API_BASE_URL = "https://telegraph.yandex.net/api/v3"

    def get_cucm_translation(self):
        result = self._request(self.API_BASE_URL + "/cucm/translation_pattern")
        return result["result"]

    def update_cucm_translation_pattern(self, pattern, destination):
        body = json.dumps({"calledPartyTransformationMask": destination})
        result = self._request(self.API_BASE_URL + "/cucm/translation_pattern/pattern/" + pattern, method="PUT", data=body)
        return not bool(result["error"])


class ImagesAssignDutyPhone(sdk2.Task):
    """
        Назначает телефон дежурного на общий номер
    """

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 360

        abc_service_id = sdk2.parameters.Integer("ABC service with duty schedule", required=True, default=DEFAULT_ABC_SERVICE_ID)
        duty_login = sdk2.parameters.String("Duty login (empty for getting from calendar)", required=False)
        reset_flag = sdk2.parameters.Bool("Reset phone", default=False)

        with sdk2.parameters.Group("Vault options"):
            vault_name = sdk2.parameters.String("Token name", default='robot_imgdevops_token')

    def on_execute(self):
        with self.memoize_stage.init:
            token = sdk2.Vault.data(self.Parameters.vault_name)
            abc_api = ABC(token)
            staff_api = Staff(token)
            telegraph_api = Telegraph(token)

        if self.Parameters.reset_flag:
            if self._update_phone(telegraph_api):
                logging.info("Phone reset")
                return
            else:
                raise TaskFailure("Can't reset duty phone")

        duty_login = self.Parameters.duty_login
        if not duty_login:
            duty_login = self._get_duty_from_abc(abc_api, self.Parameters.abc_service_id)
        if not duty_login:
            raise TaskFailure("Can't get duty login from calendar event")

        duty_info = self._get_user_info(staff_api, duty_login)
        if duty_info is None:
            raise TaskFailure("Can't find duty '{}'".format(duty_login))
        if self._update_phone(telegraph_api, duty_info["work_phone"]):
            logging.info("Phone updated to '%s' for duty '%s'", duty_info["work_phone"], duty_login)
        else:
            raise TaskFailure("Can't redirect duty phone")

    def _get_duty_from_abc(self, abc_api, abc_service_id):
        date = datetime.now().strftime("%Y-%m-%d")
        devops_shift = abc_api.get_duty_logins(abc_service_id, date)
        return devops_shift['first_line']

    def _get_user_info(self, staff_api, login):
        return staff_api.get_person_info(login, ["uid", "work_phone"])

    def _update_phone(self, telegraph_api, phone=None):
        tp = telegraph_api.get_cucm_translation()
        if tp is None:
            return False
        pattern = tp[0]["pattern"]
        telegraph_api.update_cucm_translation_pattern(pattern, phone or pattern)
        return True
