# -*- coding: utf-8 -*-

import logging
import requests

from datetime import datetime, timedelta
from simplejson.decoder import JSONDecodeError

from sandbox import sdk2

from sandbox.common.errors import TaskFailure

_LOG = logging.getLogger(__name__)

DEFAULT_CALENDAR_ID = 42622


class RestApi(object):
    def _get_request(self, operation, params=None, headers=None, api_timeout=120.0, exception=TaskFailure):
        # prepare request
        url = self.API_URL + operation
        if headers is None:
            headers = {
                'Accept': 'application/json',
                'Accept-Encoding': 'gzip, identity'
            }
            if hasattr(self, 'token'):
                headers['Authorization'] = 'OAuth {}'.format(self.token)

        # make request
        try:
            r = requests.get(url, params=params, headers=headers, timeout=api_timeout)
            r.raise_for_status()
        except requests.exceptions.Timeout:
            msg = 'API is silent'
            raise exception(msg)
        except requests.exceptions.RequestException as err:
            raise exception(err.message)

        # dump result
        try:
            return r.json()
        except JSONDecodeError:
            raise exception('Unable to parse json answer')


class CalendarApi(RestApi):
    API_URL = 'http://calendar-api.tools.yandex.net/internal'
    API_TIME_FMT = '%Y-%m-%dT%H:%M:%S'

    def _get_event(self, params):
        operation = '/get-events'
        return self._get_request(operation, params)

    def _find_longest_event(self, events):
        res_idx, max_duration = None, timedelta(seconds=0)
        for idx, event in enumerate(events):
            start, end = event['startTs'], event['endTs']
            duration = datetime.strptime(end, self.API_TIME_FMT) - datetime.strptime(start, self.API_TIME_FMT)
            if duration > max_duration:
                max_duration = duration
                res_idx = idx
        return res_idx

    def get_duty(self, calendar_id):
        logging.debug("Get duty login from {} calendar".format(calendar_id))
        current_time = datetime.now()
        finish_duty_time = current_time + timedelta(days=7 - current_time.weekday())
        params = {
            'from': current_time.strftime(self.API_TIME_FMT),
            'to': finish_duty_time.strftime(self.API_TIME_FMT),
            'tz': 'Europe/Moscow',
            'layerId': calendar_id
        }
        body = self._get_event(params)
        if 'events' not in body or len(body['events']) < 1:
            return None

        events = body['events']
        if len(events) > 1:
            # take the longest event
            idx = self._find_longest_event(events)
            event = events[idx]
        else:
            event = events[0]

        if 'attendees' not in event or len(event['attendees']) < 1:
            return None
        logging.debug("Got {} duties from {} calendar".format(events['attendees'], calendar_id))
        return event['attendees'][0]['login']


class TelegraphApi(RestApi):
    API_URL = 'https://telegraph.yandex.net/api/v3'

    def redirect_number(self, src, dst):
        logging.debug("Redirect calls from {} to {}".format(src, dst))
        operation = 'cucm/translation_pattern/pattern/{}'.format(src)
        params = {
            'calledPartyTransformationMask': dst
        }

        result = self._get_request(operation, params)
        logging.debug("Got telegraph API result: {}".format(result))
        return result.get('result', False)

    def __init__(self, token):
        self.token = token


class StaffApi(RestApi):
    API_URL = 'https://staff-api.yandex-team.ru/v3'

    def get_internal_number(self, logins):
        operation = '/persons'
        params = {
            'login': ','.join(logins),
            '_fields': 'work_phone'
        }
        logging.debug("Getting internal numbers for {}".format(logins))
        results = self._get_request(operation, params).get('result', None)
        if not results:
            logging.debug("Got empty result from staff api")
            return None
        logging.debug("Got internal numbers for {}: {}".format(logins, results))
        return [x['work_phone'] for x in results]

    def __init__(self, token):
        self.token = token


class ImagesSetDevopsDial(sdk2.Task):
    """
        Вешаем телефон для алертов на актуального дежурного
    """

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 360

        calendar_id = sdk2.parameters.Integer("Calendar id with duty schedule",
                                              required=True,
                                              default=DEFAULT_CALENDAR_ID)
        clean_redial = sdk2.parameters.Bool("Clean redirect dials", default=False)
        robot_login = sdk2.parameters.String("Robot login", required=True, default="robot-images")
        duty_login = sdk2.parameters.String("Duty login (empty for getting from calendar)", required=False)

        with sdk2.parameters.Group("Vault options"):
            vault_name = sdk2.parameters.String("Token name", default='redial_token')

    def on_execute(self):
        with self.memoize_stage.init:
            token = sdk2.Vault.data(self.Parameters.vault_name)
            staff_api = StaffApi(token=token)
            telegraph_api = TelegraphApi(token=token)

        if self.Parameters.clean_redial:
            robot_number = staff_api.get_internal_number((self.Parameters.robot_login, ))
            if not telegraph_api.redirect_number(robot_number, robot_number):
                raise TaskFailure("Something went wrong")
            return

        if self.Parameters.duty_login is None or len(self.Parameters.duty_login) < 1:
            calendar_api = CalendarApi()
            duty_login = calendar_api.get_duty(self.Parameters.calendar_id)
        else:
            duty_login = self.Parameters.duty_login

        duty_number, robot_number = staff_api.get_internal_number((duty_login, self.Parameters.robot_login))

        if not telegraph_api.redirect_number(robot_number, duty_number):
            raise TaskFailure("Something went wrong")
