import logging
import os
from sandbox import sdk2
from sandbox.projects.common.dolbilka import resources as dolbilka_resources
from sandbox.projects.images.common import SearchTaskCommonParameters
from sandbox.projects.images.embedding import EMBEDDING_STORAGE
from sandbox.projects.images.embedding.task import EDaemonComponentTask
from sandbox.projects.images.inverted_index.task import PDaemonComponentTask
from sandbox.projects.images import resource_types as images_resource_types
from sandbox.projects import resource_types
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sdk2.helpers import subprocess


class ImagesGetEmbeddingStorageResponses(sdk2.Task,
                                         EDaemonComponentTask,
                                         PDaemonComponentTask):
    """
        Get embedding storage responses
    """

    class Context(sdk2.Context):
        __first_run = True

    class Parameters(sdk2.Parameters):
        custom_settings = sdk2.parameters.Bool("Use custom partition configuration", default_value=False)
        index_state = SearchTaskCommonParameters.Parameters.index_state()
        partition_number = SearchTaskCommonParameters.Parameters.partition_number()
        search_l1_models_archive = SearchTaskCommonParameters.Parameters.search_l1_models_archive()
        edaemon_parameters = EDaemonComponentTask.Parameters()
        pdaemon_parameters = PDaemonComponentTask.Parameters()

        dexecutor = sdk2.parameters.Resource('d-executor executable',
                                             resource_type=dolbilka_resources.DEXECUTOR_EXECUTABLE)
        sdr_compare = sdk2.parameters.Resource('Search daemon compare executable',
                                               resource_type=images_resource_types.IMAGES_SDR_COMPARE_EXECUTABLE)
        plan = sdk2.parameters.Resource('Embedding storage requests (d-executor plan)',
                                        resource_type=resource_types.BASESEARCH_PLAN,
                                        required=True)

    def on_enqueue(self):
        if self.Context.__first_run is False:
            return

        if not self.Parameters.custom_settings:
            self.Parameters.pdaemon_parameters.shards_number = range(EMBEDDING_STORAGE.DEFAULT_SOURCES_NUM)
            self.Parameters.pdaemon_parameters.use_entire_partition = False

        PDaemonComponentTask.on_enqueue(self)
        EDaemonComponentTask.on_enqueue(self)

        if self.Parameters.dexecutor is None:
            self.Parameters.dexecutor = sdk2.Resource.find(type=dolbilka_resources.DEXECUTOR_EXECUTABLE,
                                                           attrs=dict(released='stable'),
                                                           state='READY').first()

        if self.Parameters.sdr_compare is None:
            self.Parameters.sdr_compare = sdk2.Resource.find(type=images_resource_types.IMAGES_SDR_COMPARE_EXECUTABLE,
                                                             attrs=dict(released='stable'),
                                                             state='READY').first()

        self.Context.__first_run = False

    def init_resources(self):
        if self.Parameters.dexecutor is None:
            raise SandboxTaskFailureError("Released stable \"d-executor executable\" not found")
        if self.Parameters.sdr_compare is None:
            raise SandboxTaskFailureError("Released stable \"Search daemon compare executable\" not found")

    def on_execute(self):
        PDaemonComponentTask.init_resources(self)
        EDaemonComponentTask.init_resources(self)
        self.init_resources()

        PDaemonComponentTask.on_execute(self)
        EDaemonComponentTask.on_execute(self)

        sdr_compare_executable = str(sdk2.ResourceData(self.Parameters.sdr_compare).path)
        dexecutor_executable = str(sdk2.ResourceData(self.Parameters.dexecutor).path)
        plan = str(sdk2.ResourceData(self.Parameters.plan).path)
        responses_main = str(self.path("embedding_responses.bin"))
        responses_control = str(self.path("embedding_responses_control.bin"))

        # First run d-executor
        process_log = sdk2.helpers.ProcessLog(self, logger=logging.getLogger("d-executor main"))
        dexecutor_main_pid = subprocess.Popen([dexecutor_executable,
                                               "-p", plan,
                                               "-P", str(EMBEDDING_STORAGE.DEFAULT_PORT),
                                               "-H", "localhost",
                                               "-m", "plan",
                                               '-s', str(EMBEDDING_STORAGE.DEFAULT_SIMULTANEOUS_REQUESTS),
                                               "-d", "--output-lenval32", "-o", responses_main], stdout=process_log.stdout, stderr=process_log.stdout)
        dexecutor_main_pid.communicate()
        if dexecutor_main_pid.returncode != 0:
            raise SandboxTaskFailureError("d-executor exit code is {}".format(dexecutor_main_pid.returncode))
        logging.info("d-executor main finished")

        # Run second d-executor for response checking
        process_log = sdk2.helpers.ProcessLog(self, logger=logging.getLogger("d-executor control"))
        dexecutor_control_pid = subprocess.Popen([dexecutor_executable,
                                                  "-p", plan,
                                                  "-P", str(EMBEDDING_STORAGE.DEFAULT_PORT),
                                                  "-H", "localhost",
                                                  "-m", "plan",
                                                  '-s', str(EMBEDDING_STORAGE.DEFAULT_SIMULTANEOUS_REQUESTS),
                                                  "-d", "--output-lenval32", "-o", responses_control], stdout=process_log.stdout, stderr=process_log.stdout)
        dexecutor_control_pid.communicate()
        if dexecutor_control_pid.returncode != 0:
            raise SandboxTaskFailureError("d-executor exit code is {}".format(dexecutor_control_pid.returncode))
        logging.info("d-executor control finished")

        # Compare main and control responses
        diff_dir = str(self.path("diff"))
        os.mkdir(diff_dir)
        process_log = sdk2.helpers.ProcessLog(self, logger=logging.getLogger("sdr_compare"))
        sdr_compare_pid = subprocess.Popen([sdr_compare_executable,
                                            "-m", "EEmbeddingStorage",
                                            "-o", diff_dir,
                                            responses_main,
                                            responses_control], stdout=process_log.stdout, stderr=process_log.stdout)
        sdr_compare_pid.communicate()
        if sdr_compare_pid.returncode != 0:
            resource = sdk2.ResourceData(images_resource_types.IMAGES_SEARCH_DAEMON_COMPARE_RESULT(self,
                                                                                                   "unstable search daemom diff",
                                                                                                   diff_dir))
            resource.ready()
            raise SandboxTaskFailureError("Embedding storage daemon is unstable (responses differs for same requests)")

        resource = sdk2.ResourceData(images_resource_types.IMAGES_SEARCH_DAEMON_RESPONSES(self,
                                                                                          "embedding storage responses",
                                                                                          responses_main,
                                                                                          daemon_type='embedding_storage'))
        resource.ready()
