import os
from sandbox import sdk2
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.images import resource_types as images_resource_types
from sandbox.projects.images.embedding import EMBEDDING_STORAGE
from sandbox.projects.images.inverted_index import INVERTED_INDEX
from sandbox.projects.images.models import resources as models_resources
from sandbox.projects.images import util
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sdk2.helpers import subprocess


class EDaemonComponentTask:
    """
        Abstract task for P-daemon running
    """

    edaemon_process_id = []
    embedding_database = None

    class Context(sdk2.Task.Context):
        __first_run = True
        edaemon_ready = None
        embedding_database_ctx = None

    class Parameters(sdk2.Task.Parameters):
        # Embedding storage executable
        embedding_storage_executable = sdk2.parameters.Resource('Embedding storage executable',
                                                                resource_type=images_resource_types.IMAGES_EMBEDDING_STORAGE_EXECUTABLE,
                                                                required=False)

        # Direct setup E-daemon shard
        embedding_database = sdk2.parameters.Resource('Embedding shard: use it for manually builded shards',
                                                      resource_type=images_resource_types.IMAGES_EMBEDDING_DATABASE,
                                                      required=False)

        # Run P-daemon(s) locally or standalone (on other sandbox host)
        use_standalone_pdaemon = sdk2.parameters.Bool('Run standalone P-daemon(s) (on other sandbox host)',
                                                      default=False,
                                                      required=True)

    def on_enqueue(self):
        if self.Context.__first_run is False:
            return

        shard_name = EDaemonComponentTask.get_shard_name(self, self.Parameters.index_state, self.Parameters.partition_number)

        if self.Parameters.embedding_storage_executable is None:
            self.Parameters.embedding_storage_executable = sdk2.Resource.find(type=images_resource_types.IMAGES_EMBEDDING_STORAGE_EXECUTABLE,
                                                                              attrs=dict(released='stable'),
                                                                              state='READY').first()
        if self.Parameters.search_l1_models_archive is None:
            self.Parameters.search_l1_models_archive = sdk2.Resource.find(type=models_resources.IMAGES_L1_DYNAMIC_MODELS_ARCHIVE,
                                                                          attrs=dict(released='stable'),
                                                                          state='READY').first()

        if self.Parameters.embedding_database is None:
            self.Parameters.embedding_database = sdk2.Resource.find(type=images_resource_types.IMAGES_EMBEDDING_DATABASE,
                                                                    attrs=dict(shard_instance=shard_name),
                                                                    state='READY').first()

        if self.Parameters.embedding_database is None:
            self.Context.embedding_database_ctx = ("shard", (media_settings.INDEX_EMBEDDING, shard_name))
        else:
            self.Context.embedding_database_ctx = ("resource", self.Parameters.embedding_database.id)

        self.Context.__first_run = False

    def init_resources(self):
        if self.Parameters.embedding_storage_executable is None:
            raise SandboxTaskFailureError("Released stable \"Embedding storage executable\" not found")
        if self.Parameters.search_l1_models_archive is None:
            raise SandboxTaskFailureError("Released stable \"Embedding storage models archive\" not found")

        self.Context.embedding_database_ctx = util.images_database_search(self, self.Context.embedding_database_ctx)
        self.embedding_database = util.images_database_get_resources([self.Context.embedding_database_ctx])[0]

    def on_execute(self):
        # Run standalone P-daemon if needed
        pdaemon_hosts = None
        if self.Parameters.use_standalone_pdaemon:
            SandboxTaskFailureError("Using standalone P-daemon not developed yet")
        else:
            pdaemon_hosts = ["localhost"]*INVERTED_INDEX.PARTITION_SIZE

        embedding_storage_executable = str(sdk2.ResourceData(self.Parameters.embedding_storage_executable).path)
        search_l1_models_archive = str(sdk2.ResourceData(self.Parameters.search_l1_models_archive).path)
        embedding_index_path = str(sdk2.ResourceData(self.embedding_database).path)
        instance_directory = str(self.path("embedding"))
        os.mkdir(instance_directory)

        # Create configs for each instance in separate directory
        # TODO: Use gencfg for create config template
        server_cfg_path = os.path.join(instance_directory, "server.cfg")
        server_cfg = open(server_cfg_path, "w")
        server_cfg.write("MaxQueueSize: {}\n".format(64))
        server_cfg.write("Threads: {}\n".format(32))
        server_cfg.close()

        embedding_cfg_path = os.path.join(instance_directory, "embedding.cfg")
        embedding_cfg = open(embedding_cfg_path, "w")
        embedding_cfg.write("InvertedIndexStorageSourceOptions {\n")
        embedding_cfg.write("    AllowConnStat: true\n")
        embedding_cfg.write("    TimeOut: \"10000ms\"\n")
        embedding_cfg.write("\n")
        embedding_cfg.write("    TaskOptions {\n")
        embedding_cfg.write("        ConnectTimeouts: \"1000ms\"\n")
        embedding_cfg.write("        SendingTimeouts: \"1000ms\"\n")
        embedding_cfg.write("    }\n")
        embedding_cfg.write("\n")
        embedding_cfg.write("    BalancingOptions {\n")
        embedding_cfg.write("        AllowDynamicWeights: false\n")
        embedding_cfg.write("        EnableIpV6: true\n")
        embedding_cfg.write("        MaxAttempts: 3\n")
        embedding_cfg.write("        RandomGroupSelection: true\n")
        embedding_cfg.write("    }\n")
        embedding_cfg.write("\n")
        embedding_cfg.write("    HedgedRequestOptions {\n")
        embedding_cfg.write("        HedgedRequestRateThreshold: 0.3\n")
        embedding_cfg.write("        HedgedRequestTimeouts: \"3ms\"\n")
        embedding_cfg.write("        HedgedRequestTimeouts: \"8ms\"\n")
        embedding_cfg.write("    }\n")
        embedding_cfg.write("}\n")
        embedding_cfg.write("SuperMindAutoParams {\n")
        embedding_cfg.write("    StartDegradationF: 0.68\n")
        embedding_cfg.write("    EndDegradationF: 0.7\n")
        embedding_cfg.write("    MultLowerBound: 0.4\n")
        embedding_cfg.write("    AutoModeType: \"exp\"\n")
        embedding_cfg.write("}\n")
        embedding_cfg.write("\n")
        embedding_cfg.write("IndexDir: \"{}\"\n".format(embedding_index_path))
        embedding_cfg.close()

        replics_cfg_path = os.path.join(instance_directory, "replics.cfg")
        replics_cfg = open(replics_cfg_path, "w")
        for i in range(INVERTED_INDEX.PARTITION_SIZE):
            replics_cfg.write("InvertedIndexStorageShardsReplics {\n")
            replics_cfg.write("  SearchScriptCgis: \"http2://{}:{}/inverted_index_storage\"\n".format(pdaemon_hosts[i], INVERTED_INDEX.DEFAULT_PORT + i))
            replics_cfg.write("}\n")
        replics_cfg.close()

        # Start E-daemon process and put process id to context
        stdout = open(os.path.join(instance_directory, "daemon.out"), "w")
        stderr = open(os.path.join(instance_directory, "daemon.err"), "w")

        self.edaemon_process_id = subprocess.Popen([embedding_storage_executable,
                                                    "--port", str(EMBEDDING_STORAGE.DEFAULT_PORT),
                                                    "--server-config", server_cfg_path,
                                                    "--embedding-storage-config", embedding_cfg_path,
                                                    "--models-archive", search_l1_models_archive,
                                                    "--embedding-storage-replics-config", replics_cfg_path], stdout=stdout, stderr=stderr)

        util.wait_for_port_open("localhost", EMBEDDING_STORAGE.DEFAULT_PORT, 7200)
        self.Context.edaemon_ready = True
        self.Context.save()

    def get_shard_name(self, index_state, partition_number):
        return "imgsembedding-{:03d}-{}".format(partition_number, index_state)
