import logging
import os
from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common import network
from sandbox.projects.images import resource_types as images_resource_types
from sandbox.projects.images import util
from sandbox.projects.images.models import resources as models_resources
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sdk2.helpers import subprocess


INTSEARCH_CONFIG_NAME = "images-sandbox-int-l1.cfg"
INTSEARCH_PORT = 25395
INTSEARCH_EVENTLOG = "intsearch-eventlog"
PARTITION_SIZE = 18


class IntSearchTask:
    """
        Abstract task for images intsearch L1 running
    """

    intsearch_process_id = None
    intsearch_eventlog_path = None

    class Context(sdk2.Task.Context):
        intsearch_first_run = True

    class Parameters(sdk2.Task.Parameters):
        # Intsearch executable
        # TODO: add int L1 to images middlesearch acceptance and create separate resource type
        intsearch_executable = sdk2.parameters.Resource('Intsearch executable',
                                                        resource_type=images_resource_types.IMAGES_L1_INTSEARCH_EXECUTABLE,
                                                        required=False)
        # Other
        intsearch_keep_eventlog = sdk2.parameters.Bool('Create resource with intsearch eventlog file',
                                                       default=False,
                                                       required=True)

    def on_enqueue(self):
        if self.Context.intsearch_first_run is False:
            return

        if self.Parameters.intsearch_executable is None:
            self.Parameters.intsearch_executable = sdk2.Resource.find(type=images_resource_types.IMAGES_L1_INTSEARCH_EXECUTABLE,
                                                                      attrs=dict(released='stable'),
                                                                      state='READY').first()
        if self.Parameters.config is None:
            self.Parameters.config = sdk2.Resource.find(type=resource_types.CONFIG_GENERATOR_CONFIGS,
                                                        attrs=dict(released='stable'),
                                                        state='READY').first()
        if self.Parameters.search_l1_models_archive is None:
            self.Parameters.search_l1_models_archive = sdk2.Resource.find(type=models_resources.IMAGES_L1_DYNAMIC_MODELS_ARCHIVE,
                                                                          attrs=dict(released='stable'),
                                                                          state='READY').first()
        self.Context.intsearch_first_run = False

    def init_resources(self):
        if self.Parameters.intsearch_executable is None:
            raise SandboxTaskFailureError("Released stable \"Intsearch executable\" not found")
        if self.Parameters.config is None:
            raise SandboxTaskFailureError("Released stable \"Generated configs by gencfg\" not found")
        if self.Parameters.search_l1_models_archive is None:
            raise SandboxTaskFailureError("Released stable \"Embedding storage models archive\" not found")

    def on_execute(self, embedding_storage_host, basesearch_shards_num, basesearch_hosts):
        logging.info("IntSearchTask started at {}".format(network.get_my_ipv6()))

        instance_directory = str(self.path("intsearch"))
        intsearch_executable = str(sdk2.ResourceData(self.Parameters.intsearch_executable).path)
        search_l1_models_archive = str(sdk2.ResourceData(self.Parameters.search_l1_models_archive).path)
        intsearch_gencfg_archive_path = str(sdk2.ResourceData(self.Parameters.config).path)
        intsearch_config_path = os.path.join(instance_directory, INTSEARCH_CONFIG_NAME)
        self.intsearch_eventlog_path = os.path.join(instance_directory, INTSEARCH_EVENTLOG)

        try:
            os.mkdir(instance_directory)
        except OSError:
            pass

        # Create intsearch config
        # Unpack gencfg archive
        # 7zr e -y generator.configs.7z "all/images-sandbox-int.cfg"
        process_log = sdk2.helpers.ProcessLog(self, logger=logging.getLogger("7zr"))
        unpack_pid = subprocess.Popen(["7zr",
                                       "e",
                                       "-o{}".format(instance_directory),
                                       intsearch_gencfg_archive_path,
                                       "all/{}".format(INTSEARCH_CONFIG_NAME)], stdout=process_log.stdout, stderr=process_log.stdout)
        unpack_pid.communicate()
        if unpack_pid.returncode != 0:
            raise Exception("Error", "7zr exit code is {}".format(unpack_pid.returncode))

        # Patch config, replace template parameters
        with open(intsearch_config_path, 'r') as cfg_file:
            cfg_file_data = cfg_file.read()
        cfg_file_data = cfg_file_data.replace('SandboxSearchSource', 'SearchSource')
        cfg_file_data = cfg_file_data.replace('SandboxAuxSource', 'AuxSource')

        # Replace host names
        host_map = dict(zip(basesearch_shards_num, basesearch_hosts))
        logging.info("Using this dictionary for intsearch config patch: {}".format(host_map))
        for i in range(PARTITION_SIZE):
            template = '${{SANDBOX_BASESEARCH_HOST_{:d}}}'.format(i)
            if i in basesearch_shards_num:
                value = host_map[i]
            else:
                value = "localhost:1"
            cfg_file_data = cfg_file_data.replace(template, value)

        cfg_file_data = cfg_file_data.replace('${SANDBOX_EMBEDDING_HOST}', embedding_storage_host)
        with open(intsearch_config_path, 'w') as cfg_file:
            cfg_file.write(cfg_file_data)

        # Make resource from int config
        intsearch_config_resource = sdk2.ResourceData(resource_types.MIDDLESEARCH_CONFIG(self,
                                                                                         self.Parameters.description,
                                                                                         intsearch_config_path,
                                                                                         daemon='intsearch'))
        intsearch_config_resource.ready()

        # Run intsearch
        stdout = open(os.path.join(instance_directory, "daemon.out"), "w")
        stderr = open(os.path.join(instance_directory, "daemon.err"), "w")
        self.intsearch_process_id = subprocess.Popen([intsearch_executable,
                                                      "-p", "{}".format(INTSEARCH_PORT),
                                                      "-d", intsearch_config_path,
                                                      "-V", "EventLog={}".format(self.intsearch_eventlog_path),
                                                      "-V", "MXNetFile={}".format(search_l1_models_archive)], cwd=instance_directory, stdout=stdout, stderr=stderr)

        util.wait_for_port_open('localhost', INTSEARCH_PORT, 7200)

    def create_eventlog_resource(self):
        self.intsearch_process_id.kill()
        self.intsearch_process_id.communicate()
        if self.Parameters.intsearch_keep_eventlog is True:
            intsearch_eventlog_resource = sdk2.ResourceData(resource_types.EVENTLOG_DUMP(self,
                                                                                         self.Parameters.description,
                                                                                         self.intsearch_eventlog_path,
                                                                                         daemon='intsearch',
                                                                                         index_state=self.Parameters.index_state))
            intsearch_eventlog_resource.ready()
