import os
from sandbox import sdk2
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.images import resource_types as images_resource_types
from sandbox.projects.images import util
from sandbox.projects.images.inverted_index import INVERTED_INDEX
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sdk2.helpers import subprocess


class PDaemonComponentTask:
    """
        Abstract task for P-daemon running
    """
    inverted_index_shards = []
    pdaemon_process_ids = []

    class Context(sdk2.Task.Context):
        __first_run = True
        __shards_search = False
        pdaemon_ready = None
        inverted_index_shards_ctx = []

    class Parameters(sdk2.Parameters):
        # Inverted index executable
        inverted_index_executable = sdk2.parameters.Resource('Inverted index executable',
                                                             resource_type=images_resource_types.IMAGES_INVERTED_INDEX_EXECUTABLE,
                                                             required=False)

        # Shard identificators for auto search
        shards_number = sdk2.parameters.List('Shard numbers',
                                             value_type=sdk2.parameters.Integer,
                                             required=False)

        use_entire_partition = sdk2.parameters.Bool('Use entire partition (run separate daemon for each partition)',
                                                    default=True,
                                                    required=False)

        lock_shards_in_memory = sdk2.parameters.Bool('Lock inverted index shards in memory',
                                                     default=True,
                                                     required=False)

        # Direct setup P-daemon shards
        # NOTE: sandbox don't support 'list of resources' parameters type, need check it manually
        inverted_index_shards = sdk2.parameters.List('Inverted index shard(s): use it for manually built shards',
                                                     value_type=sdk2.parameters.Integer,
                                                     required=False)

    def on_enqueue(self):
        if self.Context.__first_run is False:
            return

        if self.Parameters.use_entire_partition is True:
            self.Parameters.shards_number = range(INVERTED_INDEX.PARTITION_SIZE)
        if self.Parameters.inverted_index_executable is None:
            self.Parameters.inverted_index_executable = sdk2.Resource.find(type=images_resource_types.IMAGES_INVERTED_INDEX_EXECUTABLE,
                                                                           attrs=dict(released='stable'),
                                                                           state='READY').first()

        if not self.Parameters.inverted_index_shards or self.Parameters.inverted_index_shards == [0]*len(self.Parameters.shards_number):
            shards = []
            self.Context.__shards_search = True
            for i in self.Parameters.shards_number:
                shard_name = PDaemonComponentTask.get_shard_name(self, i)

                resource = sdk2.Resource.find(type=images_resource_types.IMAGES_INVERTED_INDEX_DATABASE,
                                              attrs=dict(shard_instance=shard_name),
                                              state='READY').first()
                if resource is None:
                    shards.append(0)
                else:
                    shards.append(resource.id)
            self.Parameters.inverted_index_shards = shards

        shards = []
        for i in range(len(self.Parameters.shards_number)):
            if self.Parameters.inverted_index_shards[i] == 0:
                shards.append(("shard", (media_settings.INDEX_INVERTED, PDaemonComponentTask.get_shard_name(self, self.Parameters.shards_number[i]))))
            else:
                shards.append(("resource", self.Parameters.inverted_index_shards[i]))
        self.Context.inverted_index_shards_ctx = shards

        self.Context.__first_run = False

    def init_resources(self):
        if not self.Parameters.shards_number:
            raise SandboxTaskFailureError("Empty inverted index shard list")
        if self.Parameters.inverted_index_executable is None:
            raise SandboxTaskFailureError("Released stable \"Inverted index executable\" not found")

        self.Context.inverted_index_shards_ctx = map(lambda x: util.images_database_search(self, x), self.Context.inverted_index_shards_ctx)
        self.inverted_index_shards = util.images_database_get_resources(self.Context.inverted_index_shards_ctx)

    def on_execute(self):
        for i in range(len(self.inverted_index_shards)):
            inverted_index_executable = str(sdk2.ResourceData(self.Parameters.inverted_index_executable).path)
            inverted_index_path = str(sdk2.ResourceData(self.inverted_index_shards[i]).path)
            inverted_index_port = INVERTED_INDEX.DEFAULT_PORT + self.Parameters.shards_number[i]
            instance_directory = str(self.path("inverted_index_{}".format(self.Parameters.shards_number[i])))
            os.mkdir(instance_directory)

            # Create configs for each instance in separate directory
            # TODO: Use gencfg for create config template
            server_cfg_path = os.path.join(instance_directory, "server.cfg")
            server_cfg = open(server_cfg_path, "w")
            server_cfg.write("MaxQueueSize: {}\n".format(64))
            server_cfg.write("Threads: {}\n".format(8))
            server_cfg.close()

            index_cfg_path = os.path.join(instance_directory, "index.cfg")
            index_cfg = open(index_cfg_path, "w")
            index_cfg.write("MemoryMappedIo: {}\n".format(self.Parameters.lock_shards_in_memory))
            index_cfg.write("IndexDir: \"{}\"\n".format(inverted_index_path))
            index_cfg.close()

            # Start P-daemon process and put process id to context
            stdout = open(os.path.join(instance_directory, "daemon.out"), "w")
            stderr = open(os.path.join(instance_directory, "daemon.err"), "w")

            process = subprocess.Popen([inverted_index_executable,
                                        "--port", str(inverted_index_port),
                                        "--server-config", server_cfg_path,
                                        "--inverted-index-config", index_cfg_path], stdout=stdout, stderr=stderr)
            self.pdaemon_process_ids.append(process)

        # Wait for ports ready
        for i in self.Parameters.shards_number:
            util.wait_for_port_open("localhost", INVERTED_INDEX.DEFAULT_PORT+i, 7200)
        self.Context.pdaemon_ready = True
        self.Context.save()

    def get_shard_name(self, shard_number):
        return "imgsinverted-{:03d}-{:03d}-{}".format(self.Parameters.partition_number, shard_number, self.Parameters.index_state)
