import time

from sandbox.sandboxsdk import parameters

from sandbox.projects.common import utils
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.images import responses as images_responses
from sandbox.projects.images.metasearch import task as metasearch_task
from sandbox.projects.images.metasearch import components as metasearch_components


_RESOURCE_KEY = 'out_resource_id'
_STATS_KEY = 'stats'


class InitialSeedParameter(parameters.SandboxIntegerParameter):
    name = 'initial_seed'
    description = 'Initial random seed (use 0 for current time)'
    default_value = 0


class GtasParameter(parameters.SandboxStringParameter):
    name = 'randomize_gtas'
    description = 'Attributes to randomize (use "" to randomize all attributes)'


class EnableRandomizeParameter(parameters.SandboxBoolParameter):
    name = 'enable_randomize'
    description = 'Randomize output from basesearch'
    default_value = False
    sub_fields = {
        'true': [
            InitialSeedParameter.name,
            GtasParameter.name
        ]
    }


class EnableMarketParameter(parameters.SandboxBoolParameter):
    name = 'enable_market'
    description = 'Enable fake market source'
    default_value = False


class CustomBanParameter(parameters.ResourceSelector):
    name = 'custom_ban_resource_id'
    description = 'Custom ban'
    resource_type = media_settings.ImagesSettings.middlesearch_data_resources(media_settings.INDEX_MIDDLE)
    required = False


class EarlyDBSelection(parameters.SandboxBoolParameter):
    name = "early_db_selection"
    description = 'Early db selection'
    default_value = True


class RandomizerComponent:
    """Randomizer between basesearchers and middlesearch"""

    def __init__(self, *randomizers):
        self.__randomizers = randomizers

    def __enter__(self):
        for randomizer in self.__randomizers:
            randomizer.start()
            randomizer.wait()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        for randomizer in self.__randomizers:
            randomizer.stop()


class ImagesGetMiddlesearchResponses(images_responses.ResponsesTask, metasearch_task.BaseMiddlesearchTask):
    """
        Stores responses from basesearchers via middlesearch

        This task can also inserts randomizer between basesearchers and middlesearch
        to perform a fuzzy testing
    """

    type = 'IMAGES_GET_MIDDLESEARCH_RESPONSES'

    input_parameters = \
        metasearch_task.BaseMiddlesearchTask.input_parameters + \
        images_responses.ResponsesTask.input_parameters + \
        (
            EnableMarketParameter, EnableRandomizeParameter,
            GtasParameter, InitialSeedParameter, CustomBanParameter,
            EarlyDBSelection,
        )

    execution_space = 200 * 1024  # 200 Gb

    @property
    def early_database_auto_selection(self):
        return utils.get_or_default(self.ctx, EarlyDBSelection)

    def on_execute(self):
        initial_seed = self.ctx.get(InitialSeedParameter.name, InitialSeedParameter.default_value)
        if initial_seed == InitialSeedParameter.default_value:
            self.ctx[InitialSeedParameter.name] = int(time.time())
        self._ensure_custom_ban(utils.get_or_default(self.ctx, CustomBanParameter))
        metasearch_task.BaseMiddlesearchTask.on_execute(self)

    def on_enqueue(self):
        metasearch_task.BaseMiddlesearchTask.on_enqueue(self)
        self._create_responses_resource(_RESOURCE_KEY)

    def _get_intsearch(self):
        if not self.ctx[EnableRandomizeParameter.name]:
            return metasearch_task.FakeComponent()

        initial_seed = self.ctx[InitialSeedParameter.name]
        gtas = self.ctx[GtasParameter.name]
        gtas = gtas.split(",") if gtas else []

        basesearch_randomizer = metasearch_components.ImgSearchRandomizer(
            frontend_port=self._get_randomized_basesearch_port(),
            backend_port=self._get_basesearch_port(),
            gtas=gtas,
            initial_seed=initial_seed
        )
        snippetizer_randomizer = metasearch_components.ImgSearchRandomizer(
            frontend_port=self._get_randomized_snippetizer_port(),
            backend_port=self._get_snippetizer_port(),
            gtas=gtas,
            initial_seed=initial_seed
        )
        return RandomizerComponent(basesearch_randomizer, snippetizer_randomizer)

    def _get_middlesearch(self):
        if utils.get_or_default(self.ctx, EnableMarketParameter):
            basesearches = [{
                'auxsearch_type': 'MARKET_IMAGES',
                'searchers': [('localhost', self._get_market_port())],
            }]
        else:
            basesearches = []

        middlesearch = self._get_metasearch(
            metasearch_task.MIDDLESEARCH_PARAMS,
            self._get_middlesearch_port(),
            self._get_randomized_basesearch_port(),
            self._get_randomized_snippetizer_port(),
            basesearches=basesearches
        )
        if self._need_test_with_disabled_cache():
            middlesearch.disable_cache()

        return middlesearch

    def _get_market_port(self):
        return self._get_basesearch_port() + 20

    def _get_randomized_basesearch_port(self):
        basesearch_port = self._get_basesearch_port()
        if self.ctx[EnableRandomizeParameter.name]:
            basesearch_port += 10
        return basesearch_port

    def _get_randomized_snippetizer_port(self):
        snippetizer_port = self._get_snippetizer_port()
        if self.ctx[EnableRandomizeParameter.name]:
            snippetizer_port += 10
        return snippetizer_port

    def _use_components(self, *args, **kwargs):
        if utils.get_or_default(self.ctx, EnableMarketParameter):
            with metasearch_components.ImgMarketSource(self._get_market_port()):
                self.__use_components(*args, **kwargs)
        else:
            self.__use_components(*args, **kwargs)

    def __use_components(self, basesearch, snippetizer, intsearch, middlesearch):
        self._get_responses(middlesearch, _RESOURCE_KEY, _STATS_KEY)


__Task__ = ImagesGetMiddlesearchResponses
