from sandbox.projects.common import apihelpers
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import utils
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.search import database as search_database
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common.mediasearch.acceptance import database as mediasearch_database

from sandbox.projects.images import ImagesLoadBasesearchDatabase as load_task
from sandbox.projects.images.metasearch import ImagesAnalyzeMiddlesearchPerformance as analyze_task

import logging
from enum import Enum

_SHARD_NAMES = 'shard_names'
_SHARD_CACHE = 'shard_cache'
_PERFORMANCE_CACHE = 'performance_cache'
_PERFORMANCE_STATS = 'performance_stats'


class ImagesPriemkaMiddlesearchDatabase(bugbanner.BugBannerTask):
    """
        Images middle search database acceptance task
    """

    type = "IMAGES_PRIEMKA_MIDDLESEARCH_DATABASE"

    input_parameters = (
        mediasearch_database.OldShardmapParameter,
        mediasearch_database.NewShardmapParameter,
        mediasearch_database.ForceShardLoadParameter
    )

    _OLD = 'old'
    _NEW = 'new'

    def _get_tests(self):
        return ("img", "imgquick", "cbir")

    def _get_index_type_for_tier(self, tier):
        return media_settings.ImagesSettings.index_type_for_tier(tier)

    def _get_database_resource(self, index_type):
        return media_settings.ImagesSettings.basesearch_database_resource(index_type)

    def _get_middlesearch_performance_task(self, index_type):
        task_type = analyze_task.ImagesAnalyzeMiddlesearchPerformance.type
        return task_type

    def __performance_subtask(self, search_type, old_shard_id, new_shard_id):
        description = "{}, {}".format(
            media_settings.INDEX_MIDDLE,
            search_type
        )
        sub_task = self.create_subtask(
            task_type=self._get_middlesearch_performance_task(media_settings.INDEX_MIDDLE),
            input_parameters={
                analyze_task.ImagesAnalyzeMiddlesearchPerformance.BinType.name: search_type,
                analyze_task.ImagesAnalyzeMiddlesearchPerformance.OldMiddlesearchShard.name: old_shard_id,
                analyze_task.ImagesAnalyzeMiddlesearchPerformance.NewMiddlesearchShard.name: new_shard_id
            },
            description=description
        )
        return sub_task.id

    def _load_shard(self, index_type, shard_name):
        force = self.ctx[mediasearch_database.ForceShardLoadParameter.name]
        if not force:
            logging.debug('searching for database resource in sandbox storage...')

            database_resource = apihelpers.get_last_resource_with_attribute(
                self._get_database_resource(index_type),
                media_settings.SHARD_INSTANCE_ATTRIBUTE_NAME,
                shard_name
            )
            if database_resource:
                logging.debug('database resource found. Updating cache and getting resource from cache')
                return database_resource.id

        logging.debug('don\'t use database resource from cache/sandbox. Creating download subtask instead')

        sub_ctx = {
            search_database.ShardNameParameter.name: shard_name,
            load_task.IndexTypeParameter.name: index_type
        }
        sub_task = self.create_subtask(
            task_type=load_task.ImagesLoadBasesearchDatabase.type,
            description='{}, {}, {}'.format(self.descr, index_type, shard_name),
            input_parameters=sub_ctx
        )
        return sub_task.ctx[search_database.OUT_RESOURCE_KEY]

    def __get_shard_name(self, shardmap_name):
        shardmap_path = self.sync_resource(self.ctx[shardmap_name])
        with open(shardmap_path) as shardmap_file:
            for line in shardmap_file:
                tabs = line.strip().split()
                if len(tabs) <= 2:
                    continue
                if self._get_index_type_for_tier(tabs[2]) == media_settings.INDEX_MIDDLE:
                    return tabs[1]
        raise LookupError("Can't find {} shard name at shard map {}".\
                          format(media_settings.INDEX_MIDDLE, self.ctx[shardmap_name]))

    class ReportMode(Enum):
        HTML = 1
        WIKI = 2
        TEXT = 3

    def _get_perftest_subtasks(self, mode=ReportMode.HTML):
        subtask_list = self.list_subtasks(load=True)
        subtask_list = filter(lambda x: x.type == analyze_task.ImagesAnalyzeMiddlesearchPerformance.type, subtask_list)
        subtasks = []
        for task in subtask_list:
            subtask = {"Description": task.descr}
            if mode == self.ReportMode.HTML:
                subtask["Task id"] = lb.task_link(task.id)
                subtask["Status"] = "<span class='status status_{0}'>{0}</span>".format(task.status.lower())
            elif mode == self.ReportMode.WIKI:
                subtask["Task id"] = lb.task_wiki_link(task.id)
                subtask["Status"] = task.status.lower()
            else:
                subtask["Task id"] = task.id
                subtask["Status"] = task.status.lower()

            subtasks.append(subtask)
        return subtasks

    def _load_shard_maps(self):
        if _SHARD_NAMES not in self.ctx:
            shard_names = {}
            age2params = {
                self._OLD: mediasearch_database.OldShardmapParameter.name,
                self._NEW: mediasearch_database.NewShardmapParameter.name
            }
            for age, param in age2params.iteritems():
                shard_names[age] = self.__get_shard_name(param)
            self.ctx[_SHARD_NAMES] = shard_names

    def _load_shards(self):
        if _SHARD_CACHE not in self.ctx:
            shard_cache = {}
            for age, shard_name in self.ctx[_SHARD_NAMES].iteritems():
                shard_cache[age] = self._load_shard(media_settings.INDEX_MIDDLE, shard_name)
            self.ctx[_SHARD_CACHE] = shard_cache

    def _launch_perftests(self):
        if _PERFORMANCE_CACHE not in self.ctx:
            performance_cache = {}
            for test in self._get_tests():
                performance_cache[test] = self.__performance_subtask(test,
                                                                     self.ctx[_SHARD_CACHE][self._OLD],
                                                                     self.ctx[_SHARD_CACHE][self._NEW])
            self.ctx[_PERFORMANCE_CACHE] = performance_cache

    def on_execute(self):
        logging.debug("Parsing shardmap resources...")
        self._load_shard_maps()

        logging.debug("Getting database resources...")
        self._load_shards()

        logging.debug("Running performance tests...")
        self._launch_perftests()
        utils.wait_subtasks_stop()
        self.ctx[_PERFORMANCE_STATS] = self._get_perftest_subtasks(mode=self.ReportMode.WIKI)
        utils.check_subtasks_fails()

    @property
    def footer(self):
        return {"<h3>Subtask statuses</h3>": self._get_perftest_subtasks()}


__Task__ = ImagesPriemkaMiddlesearchDatabase
