import copy
import logging

from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import task

from sandbox.projects.common import utils
from sandbox.projects.common.search import components as search_components
from sandbox.projects.images import responses as images_responses
from sandbox.projects.images.metasearch import task as metasearch_task
from sandbox.projects.images.metasearch import ImagesGetMetasearchResponses as responses_task


_OLD_BASESEARCH_PARAMS = search_components.create_noweb_basesearch_params(
    component_name="old_basesearch",
    group_name="Old basesearch",
    archive_model_required=False
)
_SUBTASK1_ID = "subtask1_id"
_SUBTASK2_ID = "subtask2_id"
_STATS_ID = "stats_id"


class UpdateModeParameter(parameters.SandboxStringParameter):
    UPDATE_SEARCH = "search"
    UPDATE_SNIPPETIZER = "snippetizer"
    name = 'update_mode'
    description = 'Update mode'
    choices = [("Search", UPDATE_SEARCH), ("Snippetizer", UPDATE_SNIPPETIZER)]
    default_value = UPDATE_SEARCH


class MaxDeltaParameter(parameters.SandboxFloatParameter):
    name = 'max_delta'
    description = 'Maximum delta'
    default_value = 0.1


class ImagesTestMetasearchUpdate(task.SandboxTask):
    """
        Test partial updates of search components from metasearch point of view

        There was a problem on production when on partial update were a lot of
        empty responses due to incompatibility between old and new version
    """
    type = 'IMAGES_TEST_METASEARCH_UPDATE'

    input_parameters = \
        (UpdateModeParameter, MaxDeltaParameter) + \
        _OLD_BASESEARCH_PARAMS.params + \
        metasearch_task.BaseMetasearchTask.input_parameters + \
        images_responses.ResponsesTask.input_parameters

    @property
    def footer(self):
        if _STATS_ID not in self.ctx:
            return [{"&nbsp;": "Calculating..."}]

        header = [
            {"key": "test", "title": "Test"},
        ]
        body = {
            "test": ["#1", "#2", "Diff"],
        }

        stats = self.ctx[_STATS_ID]
        for key, data in images_responses.STATS_VARIANTS.iteritems():
            title, predicate = data
            table_key = "stats_{}".format(key)
            header.append({"key": table_key, "title": title})
            body[table_key] = ["{:0.2f}%".format(v[key] * 100.0) for v in stats]

        return {
            "header": header,
            "body": body,
        }

    def on_execute(self):
        if _SUBTASK1_ID not in self.ctx:
            self.ctx[_SUBTASK1_ID] = self.__run_task()

        if _SUBTASK2_ID not in self.ctx:
            self.ctx[_SUBTASK2_ID] = self.__run_task(update_mode=self.ctx[UpdateModeParameter.name])

        utils.check_subtasks_fails(fail_on_first_failure=True)

        stats = [
            channel.channel.sandbox.get_task(self.ctx[task_id]).ctx.get(responses_task.STATS_KEY, {})
            for task_id in (_SUBTASK1_ID, _SUBTASK2_ID)
        ]

        def _delta(v1, v2):
            return abs(v1 - v2) / v2 if v2 else 0

        stats.append({
            key: _delta(stats[0][key], stats[1][key])
            for key, _ in images_responses.STATS_VARIANTS.iteritems()
        })
        logging.info("stats={}".format(stats))
        self.ctx[_STATS_ID] = stats

        max_diff = self.ctx[MaxDeltaParameter.name]
        if max(v for v in stats[-1].itervalues()) > max_diff:
            raise errors.SandboxTaskFailureError("Too much difference in stats ({} > {})".format(stats[-1], max_diff))

    def __run_task(self, update_mode=None):
        sub_ctx = copy.deepcopy(self.ctx)
        if update_mode == UpdateModeParameter.UPDATE_SEARCH:
            self.__update_context(sub_ctx, metasearch_task.BASESEARCH_PARAMS)
        elif update_mode == UpdateModeParameter.UPDATE_SNIPPETIZER:
            self.__update_context(sub_ctx, metasearch_task.SNIPPETIZER_PARAMS)

        return self.create_subtask(
            task_type=responses_task.ImagesGetMetasearchResponses.type,
            description=self.descr,
            input_parameters=sub_ctx,
            arch=sandboxapi.ARCH_LINUX
        ).id

    def __update_context(self, ctx, parameters):
        ctx[parameters.Binary.name] = self.ctx[_OLD_BASESEARCH_PARAMS.Binary.name]
        ctx[parameters.Config.name] = self.ctx[_OLD_BASESEARCH_PARAMS.Config.name]


__Task__ = ImagesTestMetasearchUpdate
