import logging
import httplib
import random
import string
import multiprocessing
import base64

from BaseHTTPServer import HTTPServer, BaseHTTPRequestHandler
from SocketServer import ThreadingMixIn

from sandbox.projects.common.search import queries as search_queries
from sandbox.projects.common.search.components import component as components_common


class _MarketHandler(BaseHTTPRequestHandler):
    def do_GET(self):
        # XXX: this is a dependency I cannot quite build, as google.protobuf throws an TypeError, hence the import.
        # XXX: if you want to properly include it in a binary,
        # XXX: arcadia/extsearch/images/kernel/protos/market_images_offers.proto@3930648 is probably what you want
        import sandbox.projects.common.mediasearch.market_images_offers_pb2 as market_images_offers_pb2

        status = 200
        data = ""
        try:
            query_parts = self.path.split("?", 1)
            if len(query_parts) > 1:
                offer_ids = search_queries.parse_cgi_params(query_parts[1]).get("offerid", [])
            else:
                offer_ids = []
            report = market_images_offers_pb2.TImagesReport()
            for offer_id in offer_ids:
                offer_data = report.ImagesOffers.add()
                # Fill fields with some content
                for field in offer_data.DESCRIPTOR.fields:
                    if field.type == field.TYPE_STRING:
                        setattr(offer_data, field.name, field.name)
                offer_data.WareMd5 = offer_id  # Setup key field

            data = report.SerializeToString()
        except Exception as e:
            logging.error("Failed to process query {}: {}".format(self.path, e))
            status = 500
            data = ""
        finally:
            self.send_response(status)
            self.end_headers()
            self.wfile.write(data)


class _MarketServer(ThreadingMixIn, HTTPServer):
    request_queue_size = 128

    def __init__(self, port):
        HTTPServer.__init__(self, ("localhost", port), _MarketHandler)


class _RandomizerHandler(BaseHTTPRequestHandler):
    def do_GET(self):
        status = 200
        data = ""
        try:
            if self.path == "/ping":
                return
            conn = httplib.HTTPConnection("localhost", self.server.backend_port)
            conn.request("GET", self.path)
            resp = conn.getresponse()
            status = resp.status
            data = resp.read()

            if status == 200:
                data = self.__randomize_report(data)
        except Exception as e:
            logging.exception("Failed to communicate with backend: {}".format(str(e)))
            status = 502
            data = "Failed to communicate with backend"
        finally:
            self.send_response(status)
            self.end_headers()
            self.wfile.write(data)

    def __randomize_report(self, data):
        gtas = self.server.gtas

        reqid = search_queries.parse_cgi_params(self.path).get("reqid", [""])[0]
        seed = hash(reqid) + self.server.initial_seed
        random_gen = random.Random(seed)

        if random_gen.choice((True, False)):
            return

        from sandbox.projects.common.base_search_quality.tree import meta_pb2
        report = meta_pb2.TReport()
        report.ParseFromString(data)

        for grouping in report.Grouping:
            for group in grouping.Group:
                for doc in group.Document:
                    if random_gen.choice((True, False)):
                        continue
                    ai = doc.ArchiveInfo
                    for attr in ai.GtaRelatedAttribute:
                        if (gtas and attr.Key not in gtas) or (random_gen.choice((True, False))):
                            continue
                        logging.info("Randomizing {} (docId={})...".format(attr.Key, doc.DocId))

                        random_len = random_gen.randrange(100)
                        random_val = "".join(random_gen.choice(string.printable) for _ in xrange(random_len))
                        if random_gen.choice((True, False)):
                            random_val = base64.b64encode(random_val)
                        attr.Value = random_val

        return report.SerializeToString()


class _RandomizerServer(ThreadingMixIn, HTTPServer):
    def __init__(self, frontend_port, backend_port, gtas=[], initial_seed=0):
        HTTPServer.__init__(self, ("localhost", frontend_port), _RandomizerHandler)
        self.backend_port = backend_port
        self.initial_seed = initial_seed
        self.gtas = gtas


class ImgSearchRandomizer(components_common.WaitUrlComponentMixin, components_common.Component):
    """
        Proxy between middlesearch and basesearch

        Randomizes gtas in basesearch response to perform
        a fuzzy testing of middlesearch rearrange rules
    """

    def __init__(self, frontend_port, backend_port, gtas=[], initial_seed=0):
        logging.info("Creating randomizer proxy {}->{} (gtas={}, seed={})...".format(
            frontend_port, backend_port, gtas, initial_seed))

        def wrapper():
            try:
                server = _RandomizerServer(frontend_port, backend_port, gtas=gtas, initial_seed=initial_seed)
                server.serve_forever()
            except Exception as e:
                logging.info("Failed to start randomizer server: {}".format(str(e)))

        self.__process = multiprocessing.Process(target=wrapper)
        components_common.WaitUrlComponentMixin.__init__(self, url='http://localhost:{}/ping'.format(frontend_port))

    def start(self):
        logging.info("Starting randomizer proxy...")
        self.__process.start()

    def stop(self):
        logging.info("Terminating randomizer proxy...")
        self.__process.terminate()


class ImgMarketSource(components_common.WaitUrlComponentMixin, components_common.Component):
    """Fake market source

       Sends a fake document with random data for each required offer
    """

    def __init__(self, port):
        def wrapper():
            try:
                server = _MarketServer(port)
                server.serve_forever()
            except Exception as e:
                logging.info("Failed to start market pseudo source: {}".format(str(e)))

        self.__process = multiprocessing.Process(target=wrapper)
        components_common.WaitUrlComponentMixin.__init__(self, url="http://localhost:{}/foo".format(port))

    def start(self):
        logging.info("Starting market pseudo source...")
        self.__process.start()

    def stop(self):
        logging.info("Stopping market pseudo source...")
        self.__process.terminate()


class ImgMiddlesearchServant(components_common.Component):
    """
    Wrapper over images_middlesearch component to obtain apphost port
    """

    def __init__(self, component):
        self.component = component

    def start(self):
        self.component.start()

    def wait(self, *args, **kwargs):
        self.component.wait(*args, **kwargs)

    def stop(self, *args, **kwargs):
        self.component.stop(*args, **kwargs)

    @property
    def port(self):
        return int(self.component.port) + 1

    @property
    def process(self):
        return self.component.process

    def get_environment(self):
        return self.component.get_environment()

    def set_environment(self, value):
        return self.component.set_environment(value)

    @property
    def run_cmd_patcher(self):
        return self.component.run_cmd_patcher

    @run_cmd_patcher.setter
    def run_cmd_patcher(self, value):
        self.component.run_cmd_patcher = value

    def disable_cache(self):
        self.component.disable_cache()

    def get_event_log_path(self):
        return self.component.get_event_log_path()

    def replace_config_parameter(self, name, value):
        self.component.replace_config_parameter(name, value)
