import logging
import os.path
import shutil
import subprocess

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common.search import components as search_components
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common.search.eventlog import eventlog
from sandbox.projects.images import sysinfo as images_sysinfo
from sandbox.projects.images.metasearch import resources as images_metasearch_resources
from sandbox.projects.images.metasearch import components as images_metasearch_components
from sandbox.projects.images.basesearch import components as images_basesearch_components


def create_basesearch_params(use_fusion=False, component_name=None):
    return images_basesearch_components.create_imgsearch_params(
        use_fusion=use_fusion,
        component_name="basesearch" if not component_name else component_name,
        group_name="Basesearch",
    )


def create_snippetizer_params(use_fusion=False, component_name=None):
    return images_basesearch_components.create_imgsearch_params(
        use_fusion=use_fusion,
        component_name="snippetizer" if not component_name else component_name,
        group_name="Snippetizer",
    )


def create_middlesearch_params():
    return search_components.create_middlesearch_params(
        component_name="middlesearch",
        group_name="Middlesearch",
        use_int=False
    )


def create_intsearch_params():
    return search_components.create_middlesearch_params(
        component_name="intsearch",
        group_name="Int",
        use_int=False
    )

BASESEARCH_PARAMS = create_basesearch_params()
SNIPPETIZER_PARAMS = create_snippetizer_params()
MIDDLESEARCH_PARAMS = create_middlesearch_params()
INTSEARCH_PARAMS = create_intsearch_params()

BASESEARCH_PARAMS_FUSION = create_basesearch_params(use_fusion=True, component_name="basesearch fusion")
SNIPPETIZER_PARAMS_FUSION = create_snippetizer_params(use_fusion=True, component_name="snippetizer fusion")

_IGNORED_RULES = ("ImgCommercial",)  # Buggy rules with errors
_CUSTOMIZATION_DONE_KEY = "customization_is_done"

BASESEARCH_TYPE = 'IMAGES'

BASESEARCH_PARAMS_EXTRA = search_components.get_fusion_diff_from_base_params(
    BASESEARCH_PARAMS, BASESEARCH_PARAMS_FUSION)

SNIPPETIZER_PARAMS_EXTRA = search_components.get_fusion_diff_from_base_params(
    SNIPPETIZER_PARAMS, SNIPPETIZER_PARAMS_FUSION)


class BaseMetasearchTask(images_sysinfo.SysinfoTask, task.SandboxTask):
    """
        Base task that starts metasearch components with searcher and snippetizer
    """

    required_ram = 102 * 1024
    execution_space = 30 * 1024

    input_parameters = \
        MIDDLESEARCH_PARAMS.params + \
        INTSEARCH_PARAMS.params + \
        BASESEARCH_PARAMS.params + \
        SNIPPETIZER_PARAMS.params

    client_tags = ctc.Tag.LINUX_PRECISE

    # Flag to run database autoselection on 'enqueue' or 'execute' stages.
    # Some of generation tasks unable to use database autodetection
    # on early stages due to not all plan attributes ready
    early_database_auto_selection = True

    use_saas_quick = False

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)

        if self.early_database_auto_selection:
            self._ensure_search_database()

    def on_execute(self):
        self._dump_sysinfo()

        if not self.early_database_auto_selection:
            self._ensure_search_database()

        # create components
        basesearch = self.get_imgsearch_impl(
            params=BASESEARCH_PARAMS,
            port=self._get_basesearch_port()
        )
        snippetizer = self.get_imgsearch_impl(
            params=SNIPPETIZER_PARAMS,
            port=self._get_snippetizer_port()
        )

        intsearch = self._get_intsearch()
        middlesearch = self._get_middlesearch()

        # IMGDEVOPS-361
        middlesearch_env = middlesearch.get_environment()
        middlesearch_env.update({"MKL_CBWR": "COMPATIBLE"})
        middlesearch.set_environment(middlesearch_env)

        # start components and work with them
        with basesearch, snippetizer, intsearch, middlesearch:
            self._use_components(basesearch, snippetizer, intsearch, middlesearch)
        self._after_components(basesearch, snippetizer, intsearch, middlesearch)

    def get_imgsearch_impl(self, *args, **kwargs):
        if self.use_saas_quick:
            return images_basesearch_components.get_quick_imgsearch(*args, **kwargs)

        return images_basesearch_components.get_imgsearch(*args, **kwargs)

    def _ensure_search_database(self):
        media_settings.ImagesSettings.ensure_middlesearch_index(
            self,
            self._get_queries_parameter(),
            MIDDLESEARCH_PARAMS.Index
        )
        media_settings.ImagesSettings.ensure_search_database(
            self,
            self._get_queries_parameter(),
            BASESEARCH_PARAMS.Database,
        )
        media_settings.ImagesSettings.ensure_snippetizer_database(
            self,
            self._get_queries_parameter(),
            SNIPPETIZER_PARAMS.Database,
        )

    def _get_queries_parameter(self):
        raise NotImplementedError()

    def _use_components(self, basesearch, snippetizer, intsearch, middlesearch):
        raise NotImplementedError()

    def _after_components(self, basesearch, snippetizer, intsearch, middlesearch):
        pass

    def _get_intsearch(self):
        intsearch = self._get_metasearch(
            INTSEARCH_PARAMS,
            self._get_intsearch_port(),
            self._get_basesearch_port(),
            self._get_snippetizer_port(),
        )
        intsearch.disable_cache()
        return intsearch

    def _get_middlesearch(self):
        middlesearch = self._get_metasearch(
            MIDDLESEARCH_PARAMS,
            self._get_middlesearch_port(),
            self._get_intsearch_port(),
            self._get_intsearch_port(),
        )
        middlesearch.disable_cache()
        return middlesearch

    def _get_metasearch(self, metasearch_params, metasearch_port, basesearch_port, snippetizer_port, basesearches=[]):
        return search_components.get_middlesearch(
            params=metasearch_params,
            port=metasearch_port,
            basesearches=[{
                'basesearch_type': BASESEARCH_TYPE,
                'searchers': [('localhost', basesearch_port)],
                'snippetizers': [('localhost', snippetizer_port)],
            }] + basesearches
        )

    def _get_basesearch_port(self):
        return search_components.DEFAULT_BASESEARCH_PORT

    def _get_snippetizer_port(self):
        d = 5 if self.use_saas_quick else 1
        return search_components.DEFAULT_BASESEARCH_PORT + d

    def _get_intsearch_port(self):
        return search_components.DEFAULT_MIDDLESEARCH_PORT

    def _get_middlesearch_port(self):
        return search_components.DEFAULT_MIDDLESEARCH_PORT + 1


class FakeComponent:
    """Simple placeholder for intermediate metasearch component"""

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        pass


class EnableAppHostParameter(parameters.SandboxBoolParameter):
    name = "middlesearch_enable_app_host"
    description = 'Enable AppHost on middlesearch'
    group = "Experimental"
    default_value = False


class BaseMiddlesearchTask(BaseMetasearchTask):
    """
        Base task that starts middlesearch with searcher and snippetizer
    """

    input_parameters = \
        MIDDLESEARCH_PARAMS.params + BASESEARCH_PARAMS.params + BASESEARCH_PARAMS_EXTRA.params + \
        SNIPPETIZER_PARAMS.params + SNIPPETIZER_PARAMS_EXTRA.params + \
        (EnableAppHostParameter,)

    evlog_stats_key = "evlog_stats"

    def _get_intsearch(self):
        return FakeComponent()

    def _get_middlesearch(self):
        middlesearch = self._get_metasearch(
            MIDDLESEARCH_PARAMS,
            self._get_middlesearch_port(),
            self._get_basesearch_port(),
            self._get_snippetizer_port()
        )
        if utils.get_or_default(self.ctx, EnableAppHostParameter):
            middlesearch = images_metasearch_components.ImgMiddlesearchServant(middlesearch)
        return middlesearch

    def _after_components(self, basesearch, snippetizer, intsearch, middlesearch):
        stats = self._calc_evlog_stats(middlesearch.get_event_log_path())
        self.ctx[self.evlog_stats_key] = stats
        self._verify_rearrange_errors(stats)

    @staticmethod
    def _verify_rearrange_errors(stats):
        rearrange_errors = stats["rearrange_errors"]
        total_num = sum(v for k, v in rearrange_errors.iteritems())
        if total_num:
            raise errors.SandboxTaskFailureError(
                "{} errors in rules {} was detected".format(total_num, ",".join(rearrange_errors.iterkeys()))
            )

    @staticmethod
    def _calc_evlog_stats(eventlog_path):
        stats = {
            "subsource_errors": 0,
            "rearrange_errors": {},
            "removals": {},
        }
        proc = process.run_process(
            [eventlog.get_evlogdump(), "-o", "-i", "303,328,390", eventlog_path],
            stdout=subprocess.PIPE, wait=False, outputs_to_one_file=False, log_prefix="evlogdump")
        try:
            for line in proc.stdout:
                tabs = line.strip().split()
                event = tabs[2]

                if event == "SubSourceError":
                    stats["subsource_errors"] += 1
                elif event == "RearrangeError":
                    rule = tabs[3]
                    if rule not in _IGNORED_RULES:
                        stats["rearrange_errors"][rule] = stats["rearrange_errors"].get(rule, 0) + 1
                elif event == "RearrangeInfo":
                    opcode = int(tabs[3])
                    rule = tabs[4]
                    docs = int(tabs[5])
                    #  Opcodes defined here: https://a.yandex-team.ru/arc/trunk/arcadia/search/meta/rearrange/common/rule_phase.h
                    #  Even opcode means that it is state before phase. Uneven (opcode from rule_pase + 1) - after phase
                    if opcode % 2 == 0:
                        docs_before = docs
                    else:
                        stats["removals"][rule] = stats["removals"].get(rule, 0) + (docs_before - docs)
                else:
                    raise errors.SandboxTaskFailureError("Unexpected event: {}".format(event))
        finally:
            utils.terminate_process(proc)
        return stats

    def _get_resource_type_for_ensure_custom_ban(self):
        return images_metasearch_resources.IMAGES_MIDDLESEARCH_DATA

    def _ensure_custom_ban(self, custom_ban_resource_id):
        if not custom_ban_resource_id or _CUSTOMIZATION_DONE_KEY in self.ctx:
            return

        logging.info("Customing middlesearch data...")
        middlesearch_data_key = MIDDLESEARCH_PARAMS.Data.name
        custom_ban_path = self.sync_resource(custom_ban_resource_id)
        default_data_path = self.sync_resource(self.ctx[middlesearch_data_key])

        custom_data_path = self.abs_path("mmeta.data")
        paths.remove_path(custom_data_path)

        paths.copy_path(default_data_path, custom_data_path)
        paths.add_write_permissions_for_path(custom_data_path)
        if os.path.isfile(custom_ban_path):
            paths.copy_path(custom_ban_path, os.path.join(custom_data_path, "rearrange"))
        else:
            if os.path.exists(os.path.join(custom_data_path, "rearrange", os.path.basename(custom_ban_path))):
                shutil.rmtree(os.path.join(custom_data_path, "rearrange", os.path.basename(custom_ban_path)), ignore_errors=True)
            paths.copy_path(custom_ban_path, os.path.join(custom_data_path, "rearrange", os.path.basename(custom_ban_path)))

        custom_data_resource = self.create_resource(
            "{}, customized data".format(self.descr),
            custom_data_path,
            self._get_resource_type_for_ensure_custom_ban(),
            arch=sandboxapi.ARCH_ANY,
        )
        self.mark_resource_ready(custom_data_resource)
        self.ctx[middlesearch_data_key] = custom_data_resource.id
        self.ctx[_CUSTOMIZATION_DONE_KEY] = True

    def _ensure_rtindex_extra_params(self, params):
        params_getters = (
            ('StaticData', media_settings.ImagesSettings.get_rtyserver_static_data),
            ('ShardWriterConfig', media_settings.ImagesSettings.get_rtyserver_sw_config)
        )

        logging.info('Ensure rtindex extra params.')
        for name, getter_fn in params_getters:
            ctx_key = getattr(params, name).name
            if ctx_key not in self.ctx or self.ctx[ctx_key] is None:
                logging.info('Retrieving resource for {}.'.format(ctx_key))
                self.ctx[ctx_key] = getter_fn()

    def _check_rtyserver_executable_used(self, params):
        resource_binary = channel.sandbox.get_resource(self.ctx[params.Binary.name])
        return resource_binary.type == str(resource_types.IMGSEARCH_RTYSERVER_EXECUTABLE)

    def on_execute(self):
        global BASESEARCH_TYPE
        basesearch_is_rtindex = self._check_rtyserver_executable_used(BASESEARCH_PARAMS)
        snippetizer_is_rtindex = self._check_rtyserver_executable_used(SNIPPETIZER_PARAMS)
        assert basesearch_is_rtindex == snippetizer_is_rtindex, "Basesearch and Snippetizer binary types are not consistent."
        logging.info('Using rtindex basesearch = {}'.format(basesearch_is_rtindex))
        BaseMetasearchTask.use_saas_quick = basesearch_is_rtindex
        if self.use_saas_quick:
            media_settings.ImagesSettings.USE_SAAS_QUICK = True
            BASESEARCH_TYPE = 'IMAGESQUICK'
            if not search_components._check_is_fusion_params(BASESEARCH_PARAMS):
                search_components.extend_basesearch_params_to_fusion_params(BASESEARCH_PARAMS, BASESEARCH_PARAMS_EXTRA)
            if not search_components._check_is_fusion_params(SNIPPETIZER_PARAMS):
                search_components.extend_basesearch_params_to_fusion_params(SNIPPETIZER_PARAMS, SNIPPETIZER_PARAMS_EXTRA)

            self._ensure_rtindex_extra_params(BASESEARCH_PARAMS)
            self._ensure_rtindex_extra_params(SNIPPETIZER_PARAMS)

        return BaseMetasearchTask.on_execute(self)
