import logging
import os
from sandbox import sdk2
from sandbox.projects.common import network
from sandbox.projects import resource_types
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.images import util
from sandbox.projects.images.models import resources as models_resources
from sandbox.sdk2.helpers import subprocess
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


MIDDLESEARCH_CONFIG_NAME = "images-sandbox-mmeta.cfg"
MIDDLESEARCH_PORT = 9402
MIDDLESEARCH_EVENTLOG = "middleserch-eventlog"


class MiddleSearchTask:
    """
        Abstract task for images middlesearch running
    """

    middlesearch_process_id = None
    middlesearch_database = None
    middlesearch_eventlog_path = None

    class Context(sdk2.Task.Context):
        middlesearch_first_run = True
        middlesearch_database_ctx = None

    class Parameters(sdk2.Task.Parameters):
        # Images middlesearch executable
        middlesearch_executable = sdk2.parameters.Resource('Middlesearch executable',
                                                           resource_type=resource_types.IMAGES_MIDDLESEARCH_EXECUTABLE,
                                                           required=False)

        # Matrixnet models archive
        middlesearch_models_archive = sdk2.parameters.Resource('Middlesearch matrixnet models archive',
                                                               resource_type=models_resources.IMAGES_MIDDLESEARCH_DYNAMIC_MODELS_ARCHIVE,
                                                               required=False)

        # Direct setup middlesearch shard
        middlesearch_database = sdk2.parameters.Resource('Middlesearch index: use it for manually builded shards',
                                                         resource_type=resource_types.IMAGES_MIDDLESEARCH_INDEX,
                                                         required=False)

        # Other
        middlesearch_keep_eventlog = sdk2.parameters.Bool('Create resource with middlesearch eventlog file',
                                                          default=False,
                                                          required=True)

    def on_enqueue(self):
        if self.Context.middlesearch_first_run is False:
            return

        if self.Parameters.middlesearch_executable is None:
            self.Parameters.middlesearch_executable = sdk2.Resource.find(type=resource_types.IMAGES_MIDDLESEARCH_EXECUTABLE,
                                                                         attrs=dict(released='stable'),
                                                                         state='READY').first()
        if self.Parameters.config is None:
            self.Parameters.config = sdk2.Resource.find(type=resource_types.CONFIG_GENERATOR_CONFIGS,
                                                        attrs=dict(released='stable'),
                                                        state='READY').first()
        if self.Parameters.middlesearch_models_archive is None:
            self.Parameters.middlesearch_models_archive = sdk2.Resource.find(type=models_resources.IMAGES_MIDDLESEARCH_DYNAMIC_MODELS_ARCHIVE,
                                                                             attrs=dict(released='stable'),
                                                                             state='READY').first()
        if self.Parameters.middlesearch_database is None:
            self.Parameters.middlesearch_database = sdk2.Resource.find(type=resource_types.IMAGES_MIDDLESEARCH_INDEX,
                                                                       attrs=dict(shard_instance=self.get_shard_name()),
                                                                       state='READY').first()

        if self.Parameters.middlesearch_database is None:
            self.Context.middlesearch_database_ctx = ("shard", (media_settings.INDEX_MIDDLE, self.get_shard_name()))
        else:
            self.Context.middlesearch_database_ctx = ("resource", self.Parameters.middlesearch_database.id)

        self.Context.middlesearch_first_run = False

    def init_resources(self):
        if self.Parameters.middlesearch_executable is None:
            raise SandboxTaskFailureError("Released stable \"Middlesearch executable\" not found")
        if self.Parameters.config is None:
            raise SandboxTaskFailureError("Released stable \"Generated configs by gencfg\" not found")
        if self.Parameters.middlesearch_models_archive is None:
            raise SandboxTaskFailureError("Released stable \"Middlesearch matrixnet models archive\" not found")

        self.Context.middlesearch_database_ctx = util.images_database_search(self, self.Context.middlesearch_database_ctx)
        self.middlesearch_database = util.images_database_get_resources([self.Context.middlesearch_database_ctx])[0]

    def on_execute(self, int_l1_host_name):
        logging.info("MiddleSearchTask started at {}".format(network.get_my_ipv6()))

        middlesearch_executable = str(sdk2.ResourceData(self.Parameters.middlesearch_executable).path)
        middlesearch_models_archive = str(sdk2.ResourceData(self.Parameters.middlesearch_models_archive).path)
        middlesearch_index_path = str(sdk2.ResourceData(self.middlesearch_database).path)
        middlesearch_gencfg_archive_path = str(sdk2.ResourceData(self.Parameters.config).path)
        instance_directory = str(self.path("middlesearch"))
        middlesearch_config_path = os.path.join(instance_directory, MIDDLESEARCH_CONFIG_NAME)
        self.middlesearch_eventlog_path = os.path.join(instance_directory, MIDDLESEARCH_EVENTLOG)

        try:
            os.mkdir(instance_directory)
        except OSError:
            pass

        # Create middlesearch config
        # Unpack gencfg archive
        # 7zr e -y generator.configs.7z "all/images-sandbox-mmeta.cfg"
        process_log = sdk2.helpers.ProcessLog(self, logger=logging.getLogger("7zr"))
        unpack_pid = subprocess.Popen(["7zr",
                                       "e",
                                       "-o{}".format(instance_directory),
                                       middlesearch_gencfg_archive_path,
                                       "all/{}".format(MIDDLESEARCH_CONFIG_NAME)], stdout=process_log.stdout, stderr=process_log.stdout)
        unpack_pid.communicate()
        if unpack_pid.returncode != 0:
            raise Exception("Error", "7zr exit code is {}".format(unpack_pid.returncode))

        # Patch config, replace template parameters
        with open(middlesearch_config_path, 'r') as cfg_file:
            cfg_file_data = cfg_file.read()
        cfg_file_data = cfg_file_data.replace('SandboxSearchSource', 'SearchSource')
        cfg_file_data = cfg_file_data.replace('${SANDBOX_INT_L1_HOST}', int_l1_host_name)
        with open(middlesearch_config_path, 'w') as cfg_file:
            cfg_file.write(cfg_file_data)

        # Make resource from int config
        middlesearch_config_resource = sdk2.ResourceData(resource_types.MIDDLESEARCH_CONFIG(self,
                                                                                            self.Parameters.description,
                                                                                            middlesearch_config_path,
                                                                                            daemon='middlesearch'))
        middlesearch_config_resource.ready()

        # Run middlesearch
        stdout = open(os.path.join(instance_directory, "daemon.out"), "w")
        stderr = open(os.path.join(instance_directory, "daemon.err"), "w")
        self.middlesearch_process_id = subprocess.Popen([middlesearch_executable,
                                                         "-p", "{}".format(MIDDLESEARCH_PORT),
                                                         "-d", middlesearch_config_path,
                                                         "-V", "EventLog={}".format(self.middlesearch_eventlog_path),
                                                         "-V", "RearrangeIndexDir={}".format(middlesearch_index_path),
                                                         "-V", "MXNetFile={}".format(middlesearch_models_archive)], cwd=instance_directory, stdout=stdout, stderr=stderr)

        util.wait_for_port_open('localhost', MIDDLESEARCH_PORT, 7200)

    def create_eventlog_resource(self):
        self.middlesearch_process_id.kill()
        self.middlesearch_process_id.communicate()
        if self.Parameters.middlesearch_keep_eventlog is True:
            middlesearch_eventlog_resource = sdk2.ResourceData(resource_types.EVENTLOG_DUMP(self,
                                                                                            self.Parameters.description,
                                                                                            self.middlesearch_eventlog_path,
                                                                                            daemon='middlesearch',
                                                                                            index_state=self.Parameters.index_state))
            middlesearch_eventlog_resource.ready()

    def get_shard_name(self):
        return "imgmmeta-000-{}".format(self.Parameters.index_state)
