import itertools
import logging

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import apihelpers
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common import utils
from sandbox.projects.common.dynamic_models import archiver as models_archiver
from sandbox.projects.common.dynamic_models import compare as models_compare
from sandbox.projects.common.dynamic_models import download as models_download
from sandbox.projects.common.dynamic_models import matrixnet as models_matrixnet
from sandbox.projects.common.nanny import nanny
from sandbox.projects.images.resources import task as resources_task


OUT_RESOURCE_KEY = 'out_resource_id'


class IndexTypeParameter(parameters.SandboxStringParameter):
    name = 'index_type'
    description = 'Index type'
    choices = [
        ('Main', media_settings.INDEX_MAIN),
        ('Middle', media_settings.INDEX_MIDDLE),
    ]
    default_value = media_settings.INDEX_MAIN


class ModelsTypeParameter(parameters.SandboxStringParameter):
    name = 'models_type'
    description = 'Models type'
    choices = [
        ('Production', media_settings.MODELS_PRODUCTION),
        ('Experimental', media_settings.MODELS_EXPERIMENTAL),
        ('Final', media_settings.MODELS_FINAL),
    ]
    default_value = media_settings.MODELS_FINAL


class ModelsPathParameter(parameters.SandboxSvnUrlParameter):
    name = 'models_svn_url'
    description = 'SVN url for basesearch models:'
    required = False


class BetaModelsSkynetParameter(parameters.SandboxStringParameter):
    name = 'beta_models_skynet'
    description = 'Download beta models (rbtorrent)'
    multiline = False


class BetaModelsFmlParameter(parameters.SandboxStringParameter):
    name = 'beta_models_links'
    description = 'Download beta models (file_name : url)'
    multiline = True


class ImagesBuildDynamicModels(nanny.ReleaseToNannyTask, resources_task.ImagesProductionResourcesTask, task.SandboxTask):
    """
        Retrieve models from specified source and build archive
    """

    type = 'IMAGES_BUILD_DYNAMIC_MODELS'
    client_tags = ctc.Tag.LINUX_PRECISE
    input_parameters = (
        IndexTypeParameter,
        ModelsTypeParameter,
        ModelsPathParameter,
        BetaModelsSkynetParameter,
        BetaModelsFmlParameter,
    )
    environment = (environments.SvnEnvironment(), )

    @property
    def footer(self):
        return models_compare.generate_diff_footer(self.ctx.get('diff'))

    @property
    def nanny_token(self):
        return self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token')

    def on_release(self, additional_parameters):
        models_type = utils.get_or_default(self.ctx, ModelsTypeParameter)
        if models_type == media_settings.MODELS_FINAL:
            logging.info("Final models. Sending to nanny")
            nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        else:
            logging.info("Intermediate models. Skipping nanny")
            task.SandboxTask.on_release(self, additional_parameters)

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)

        self.ctx[OUT_RESOURCE_KEY] = self.create_resource(
            self.descr,
            self._get_models_archive_path(),
            self._get_models_resource_type(),
            arch=sandboxapi.ARCH_ANY
        ).id

    def on_execute(self):
        self._create_models_archive()
        self._compare_models_archive()

    def _create_models_archive(self):
        mxops_path = self.sync_resource(self._get_mx_ops_executable())
        archiver_path = self.sync_resource(self._get_archiver_executable())

        # export models from repository
        models_dir = self.path('models')
        paths.remove_path(models_dir)

        svn_url = self.ctx[ModelsPathParameter.name]
        if not svn_url:
            self.ctx[ModelsPathParameter.name] = svn_url = media_settings.ImagesSettings.models_url(
                utils.get_or_default(self.ctx, IndexTypeParameter),
                utils.get_or_default(self.ctx, ModelsTypeParameter)
            )
        svn_revision = svn.Arcadia.log(svn_url, "HEAD", 1, limit=1)[0]["revision"]
        svn.Arcadia.export(svn_url, models_dir, revision=svn_revision)

        # download additional models
        skynet_dir = self.path('models_skynet')
        paths.make_folder(skynet_dir, delete_content=True)
        skynet_models = models_download.sky(skynet_dir, self.ctx[BetaModelsSkynetParameter.name])

        fml_dir = self.path('models_fml')
        paths.make_folder(fml_dir, delete_content=True)
        fml_urls = models_download.parse_fml_urls(self.ctx[BetaModelsFmlParameter.name])
        fml_models = models_download.download_urls(fml_dir, fml_urls)

        for model in itertools.chain(skynet_models, fml_models):
            paths.copy_path(model, models_dir)

        # ensure each model contain formula-id field (if not, then set it as md5)
        models_matrixnet.ensure_models_id(mxops_path, [models_dir])

        models_matrixnet.ensure_model_slices(mxops_path, [models_dir], 'images_production')

        # create models archive
        models_archiver.create(archiver_path, self._get_models_archive_path(), False, models_dir)
        channel.sandbox.set_resource_attribute(self.ctx[OUT_RESOURCE_KEY], "revision", str(svn_revision))

    def _compare_models_archive(self):
        released_archive = apihelpers.get_last_released_resource(
            self._get_models_resource_type(),
            arch=sandboxapi.ARCH_ANY
        )
        if not released_archive:
            return

        diff = models_compare.compare_archives(
            self.sync_resource(self._get_archiver_executable()),
            self.sync_resource(released_archive.id),
            self._get_models_archive_path()
        )
        # Note: previous method uses generators as a dict values
        diff = dict((field, sorted(diff[field])) for field in diff)
        self.ctx['diff'] = diff

    def _get_models_archive_path(self):
        return self.abs_path('models.archive')

    def _get_models_resource_type(self):
        index_type = utils.get_or_default(self.ctx, IndexTypeParameter)
        models_type = utils.get_or_default(self.ctx, ModelsTypeParameter)
        return media_settings.ImagesSettings.models_resource(index_type, models_type)


__Task__ = ImagesBuildDynamicModels
