# -*- coding: utf-8 -*-

from sandbox.projects.common.dynamic_models.compare import compare_archives, generate_diff_footer
from sandbox.projects.common.utils import sandbox_resource
from sandbox.projects.common.search.settings import ImagesSettings
from sandbox.projects.images.models import resources as images_models_resources

from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class Archive1Parameter(ResourceSelector):
    """
        Первый архив для сравнения
    """

    name = 'archive_1'
    description = 'Archive 1'

    resource_type = images_models_resources.IMAGES_DYNAMIC_MODELS_ARCHIVE
    required = True


class Archive2Parameter(ResourceSelector):
    """
        Второй архив для сравнения
    """

    name = 'archive_2'
    description = 'Archive 2'

    resource_type = images_models_resources.IMAGES_DYNAMIC_MODELS_ARCHIVE
    required = True


class StrictModeParameter(SandboxBoolParameter):
    """
        Если модели изменились, то считать таск проваленным
    """

    name = 'strict_mode'
    description = 'Fail if models changed'
    default_value = False


class ImagesCompareDynamicModels(SandboxTask):
    """
        Таск сравнивает два архива с моделями и строит табличку
        с добавленными, удаленными, измененными и нетронутыми моделями
    """

    type = 'IMAGES_COMPARE_DYNAMIC_MODELS'

    input_parameters = (
        Archive1Parameter,
        Archive2Parameter,
        StrictModeParameter,
    )

    @property
    def footer(self):
        return generate_diff_footer(self.ctx.get('diff'))

    def on_execute(self):
        # get latest archiver executable
        archiver_attributes = ImagesSettings.RESOURCE_PRIEMKA_ATTRIBUTES
        archiver_path = sandbox_resource(
            resource_types.ARCHIVER_TOOL_EXECUTABLE,
            archiver_attributes[0],
            archiver_attributes[1]
        )

        # calculate difference if both parameters were set
        arch1_path = self.sync_resource(self.ctx[Archive1Parameter.name])
        arch2_path = self.sync_resource(self.ctx[Archive2Parameter.name])

        diff = compare_archives(archiver_path, arch1_path, arch2_path)
        # Note: previous method uses generators as a dict values
        diff = dict((field, sorted(diff[field])) for field in diff)
        self.ctx['diff'] = diff

        if self.ctx.get(StrictModeParameter.name, StrictModeParameter.default_value):
            full_diff = diff.get('added', []) + diff.get('deleted', []) + diff.get('changed', [])
            if full_diff:
                raise SandboxTaskFailureError('Models changed: %s' % sorted(full_diff))


__Task__ = ImagesCompareDynamicModels
