import copy
import logging
import time

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import sandboxapi

from sandbox.projects.common import apihelpers
from sandbox.projects.common.dynamic_models import archiver as models_archiver
from sandbox.projects.common.dynamic_models import compare as models_compare
from sandbox.projects.common.dynamic_models import merge as models_merge
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.common.mediasearch import release as media_release
from sandbox.projects.common.mediasearch import yasm_task
from sandbox.projects.images.basesearch import task as images_task
from sandbox.projects.images.metasearch import task as metasearch_task
from sandbox.projects.images.models import ImagesTestBasesearchModels as basesearch_task
from sandbox.projects.images.models import ImagesTestMiddlesearchModels as middlesearch_task
from sandbox.projects.images.resources import task as resources_task

from sandbox.projects.images.ImagesReleaseMiddlesearchModelsTimestamp import ImagesMiddlesearchModelsTimestamp


_BUILDER_KEY = "builder"
_DIFF_KEY = "diff"


class IndexTypeParameter(parameters.SandboxStringParameter):
    name = 'index_type'
    description = 'Index type'
    choices = [
        ('Main', search_settings.INDEX_MAIN),
        ('Middle', search_settings.INDEX_MIDDLE),
    ]
    default_value = search_settings.INDEX_MAIN


class ImagesMergeDynamicModels(yasm_task.YasmTask,
                               resources_task.ImagesProductionResourcesTask,
                               media_release.BaseNannyTask,
                               media_release.BaseReleaseTask):
    """
        Merge and release archive with dynamic models for basesearch
    """

    type = 'IMAGES_MERGE_DYNAMIC_MODELS'
    input_parameters = (IndexTypeParameter,) + media_release.BaseReleaseTask.input_parameters

    @property
    def release_subject(self):
        return "images/models-" + self.ctx[IndexTypeParameter.name] + "-{timestamp}"

    @property
    def release_comment(self):
        return "daily images models ({})".format(self.ctx[IndexTypeParameter.name])

    @property
    def release_resources(self):
        return (self.__get_models_resource(search_settings.MODELS_FINAL),)

    @property
    def push_signal_name(self):
        return "merge_{}_models".format(self.ctx[IndexTypeParameter.name])

    @property
    def footer(self):
        if _BUILDER_KEY in self.ctx:
            if _DIFF_KEY in self.ctx:
                return models_compare.generate_diff_footer(self.ctx[_DIFF_KEY])
            else:
                return "Merging..."
        else:
            return ""

    def on_execute(self):
        if _BUILDER_KEY not in self.ctx:
            return media_release.BaseReleaseTask.on_execute(self)

        self.__build_models()
        self.__compare_models()
        if self.ctx[IndexTypeParameter.name] == search_settings.INDEX_MIDDLE:
            self.__create_timestamp()

    def _skip_build(self):
        if not media_release.BaseReleaseTask._skip_build(self):
            return False

        production_resource = apihelpers.get_last_released_resource(
            self.__get_models_resource(search_settings.MODELS_PRODUCTION)
        )
        experimental_resource = apihelpers.get_last_released_resource(
            self.__get_models_resource(search_settings.MODELS_EXPERIMENTAL)
        )
        if not production_resource or not experimental_resource:
            logging.info("Nothing to build from")
            return True

        released_resource = apihelpers.get_last_released_resource(
            self.__get_models_resource(search_settings.MODELS_FINAL)
        )
        if not released_resource:
            logging.info("No released resources. One needs to be built.")
            return False

        current_production = str(production_resource.attributes.get("revision", "NONE"))
        current_experimental = str(experimental_resource.attributes.get("revision", "NONE"))
        logging.info("Current production: {}, experimental: {}".format(current_production, current_experimental))

        released_production = str(released_resource.attributes.get("production_revision", "NONE"))
        released_experimental = str(released_resource.attributes.get("experimental_revision", "NONE"))
        logging.info("Released production: {}, experimental: {}".format(released_production, released_experimental))

        return current_production == released_production and current_experimental == released_experimental

    def _do_build(self):
        sub_ctx = copy.deepcopy(self.ctx)
        sub_ctx[_BUILDER_KEY] = True
        sub_task = self.create_subtask(
            task_type=self.type,
            description=self.descr,
            input_parameters=sub_ctx,
            inherit_notifications=True,
            priority=self.priority
        )
        return sub_task.id
        self.__build_models()

    def _do_test(self, build_task_id):
        new_models = apihelpers.list_task_resources(
            build_task_id,
            self.__get_models_resource(search_settings.MODELS_FINAL)
        )
        if not new_models:
            raise errors.SandboxTaskFailureError('Failed to find new models {}'.format(self.models_resource))

        index_type = self.ctx[IndexTypeParameter.name]
        if index_type == search_settings.INDEX_MAIN:
            sub_type = basesearch_task.ImagesTestBasesearchModels.type
            sub_ctx = {
                images_task.BASESEARCH_PARAMS.Binary.name: self._get_basesearch_executable(),
                images_task.BASESEARCH_PARAMS.Config.name: self._get_basesearch_config(),
                images_task.BASESEARCH_PARAMS.Database.name: self._get_basesearch_database(),
                images_task.BASESEARCH_PARAMS.ArchiveModel.name: new_models[0].id,
                images_task.BASESEARCH_PARAMS.PoliteMode.name: False,
            }
        elif index_type == search_settings.INDEX_MIDDLE:
            sub_type = middlesearch_task.ImagesTestMiddlesearchModels.type
            sub_ctx = {
                metasearch_task.MIDDLESEARCH_PARAMS.Binary.name: self._get_middlesearch_executable(),
                metasearch_task.MIDDLESEARCH_PARAMS.Config.name: self._get_middlesearch_config(),
                metasearch_task.MIDDLESEARCH_PARAMS.Data.name: self._get_middlesearch_data(),
                metasearch_task.MIDDLESEARCH_PARAMS.Index.name: self._get_middlesearch_index(),
                metasearch_task.MIDDLESEARCH_PARAMS.ArchiveModel.name: new_models[0].id
            }
            for params in (metasearch_task.BASESEARCH_PARAMS, metasearch_task.SNIPPETIZER_PARAMS):
                sub_ctx.update({
                    params.Binary.name: self._get_basesearch_executable(),
                    params.Config.name: self._get_basesearch_config(),
                    params.Database.name: self._get_basesearch_database(),
                    params.ArchiveModel.name: self._get_basesearch_models(),
                    params.PoliteMode.name: False,
                })
        else:
            raise errors.SandboxTaskFailureError("Unsupported index type {}".format(index_type))

        sub_task = self.create_subtask(
            task_type=sub_type,
            input_parameters=sub_ctx,
            description=self.descr,
        )
        return [sub_task.id]

    def _do_monitor(self):
        # TemporaryError is not compatible with 'Sequence run' schedulers
        self._yasm_notify(restart_on_failure=False)

    def __build_models(self):
        production_models_dir, production_revision = self.__unpack_models(search_settings.MODELS_PRODUCTION)
        experimental_models_dir, experimental_revision = self.__unpack_models(search_settings.MODELS_EXPERIMENTAL)

        models_dir = self.path('models')
        paths.make_folder(models_dir, delete_content=True)
        models_merge.merge(models_dir, production_models_dir)
        models_merge.merge(models_dir, experimental_models_dir)
        models_archiver.create(
            self.sync_resource(self._get_archiver_executable()),
            self.__get_models_archive_path(),
            False,
            models_dir
        )

        self.create_resource(
            self.descr,
            self.__get_models_archive_path(),
            self.__get_models_resource(search_settings.MODELS_FINAL),
            arch=sandboxapi.ARCH_ANY,
            attributes={
                "production_revision": production_revision,
                "experimental_revision": experimental_revision,
            }
        )

    def __compare_models(self):
        released_archive = apihelpers.get_last_released_resource(
            self.__get_models_resource(search_settings.MODELS_FINAL),
            arch=sandboxapi.ARCH_ANY
        )
        if not released_archive:
            return

        diff = models_compare.compare_archives(
            self.sync_resource(self._get_archiver_executable()),
            self.sync_resource(released_archive.id),
            self.__get_models_archive_path()
        )
        # Note: previous method uses generators as a dict values
        diff = dict((field, sorted(diff[field])) for field in diff)
        self.ctx[_DIFF_KEY] = diff

    def __unpack_models(self, models_type):
        models_resource = apihelpers.get_last_released_resource(self.__get_models_resource(models_type))
        models_path = self.sync_resource(models_resource.id)
        models_dir = self.path(models_type)

        paths.make_folder(models_dir, delete_content=True)
        models_archiver.unpack(self.sync_resource(self._get_archiver_executable()), models_path, models_dir)

        return models_dir, str(models_resource.attributes.get("revision", "NONE"))

    def __create_timestamp(self):
        ts_resorce = self.create_resource(
            self.descr,
            "models_timestamp.txt",
            ImagesMiddlesearchModelsTimestamp,
            arch=sandboxapi.ARCH_ANY,
        )
        with open(ts_resorce.path, "w") as ts_file:
            ts_file.write('{}'.format(int(time.time())))

    def __get_models_resource(self, models_type):
        return search_settings.ImagesSettings.models_resource(self.ctx[IndexTypeParameter.name], models_type)

    def __get_models_archive_path(self):
        return "models.archive"


__Task__ = ImagesMergeDynamicModels
