import json

from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import task

from sandbox.projects.common import dolbilka
from sandbox.projects.common import utils
from sandbox.projects.common.differ import coloring
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.images.basesearch import ImagesAnalyzeBasesearchPerformance as analyze_task
from sandbox.projects.images.basesearch import ImagesAnalyzeRtyserverPerformance as analyze_task_rtyserver
from sandbox.projects.images.models import ImagesBuildDynamicModels as build_task
from sandbox.projects.images.resources import task as resources_task
from sandbox.projects.images.resources import ImagesGenerateRtyserverRequests as rty_generate_plan_task
from sandbox.projects.tank import offline


_BUILD_TASK_ID = 'build_task_id'
_TEST_TASK_IDS = 'test_task_ids'


class ImagesPriemkaBasesearchModels(resources_task.ImagesProductionResourcesTask, task.SandboxTask):
    """
        Builds production models and runs tests to verify them
    """

    type = 'IMAGES_PRIEMKA_BASESEARCH_MODELS'
    input_parameters = []

    @property
    def footer(self):
        items = [{
            "<h4>Build</h4>": self._build_footer()
        }, {
            "<h4>Performance tests<h4>": self._performance_footer()
        }]
        return [{"content": item} for item in items]

    def _build_footer(self):
        if _BUILD_TASK_ID not in self.ctx:
            return [{"&nbsp;": "Building..."}]

        task_id = self.ctx[_BUILD_TASK_ID]
        task = channel.channel.sandbox.get_task(task_id)
        return {
            "header": [
                {"key": "task", "title": "&nbsp;"},
                {"key": "status", "title": "&nbsp;"}
            ],
            "body": {
                "task": ["<a href='/task/{0}/view'>Build task</a>".format(task_id)],
                "status": [utils.colored_status(task.status)],
            }
        }

    def _performance_footer(self):
        if _TEST_TASK_IDS not in self.ctx:
            return [{"&nbsp;": "Calculating..."}]

        results = []
        for title, task_id in self.ctx[_TEST_TASK_IDS].iteritems():
            task = channel.channel.sandbox.get_task(task_id)
            data = {
                "title": title,
                "task_id": task.id,
                "task_status": task.status,
            }
            data.update(task.ctx.get("new_stats", {}).get('diff', {}))
            results.append(data)

        def _format(fmt, value):
            return fmt.format(value) if value is not None else "-"

        return {
            "header": [
                {"key": "test_name", "title": "Test"},
                {"key": "test_status", "title": "Status"},
                {"key": "shooting.rps_0.5", "title": "RPS P50"},
                {"key": "shooting.rps_stddev", "title": "RPS stddev"},
                {"key": "shooting.errors", "title": "Errors"},
            ],
            "body": {
                "test_name": ["<a href='/task/{}/view'>{}</a>".format(v["task_id"], v["title"]) for v in results],
                "test_status": [utils.colored_status(v["task_status"]) for v in results],
                "shooting.rps_0.5": [coloring.color_diff(v.get("shooting.rps_0.5"), max_diff=-1) for v in results],
                "shooting.rps_stddev": [_format("{:0.2f}%", v.get("shooting.rps_stddev")) for v in results],
                "shooting.errors": [_format("{}%", v.get("shooting.errors")) for v in results],
            },
        }

    def on_execute(self):
        search_settings.ImagesSettings.USE_SAAS_QUICK = True  # IMAGES-16238

        if _BUILD_TASK_ID not in self.ctx:
            old_models_resource_id = self.__get_old_models_resource_id()

            build_task_id, new_models_resource_id = self._build_models()
            self.ctx[_BUILD_TASK_ID] = build_task_id

            self.ctx[_TEST_TASK_IDS] = {
                index_type: self._test_performance(index_type, old_models_resource_id, new_models_resource_id)
                for index_type in (search_settings.INDEX_MAIN, search_settings.INDEX_QUICK)
            }

        utils.check_subtasks_fails(fail_on_first_failure=True)

    def _build_models(self):
        sub_ctx = {
            build_task.IndexTypeParameter.name: search_settings.INDEX_MAIN,
            build_task.ModelsTypeParameter.name: search_settings.MODELS_PRODUCTION,
        }
        sub_task = self.create_subtask(
            task_type=build_task.ImagesBuildDynamicModels.type,
            input_parameters=sub_ctx,
            description=self.descr,
        )
        return sub_task.id, sub_task.ctx[build_task.OUT_RESOURCE_KEY]

    def _test_performance(self, index_type, old_models_resource_id, new_models_resource_id):
        plan_id = self._get_basesearch_plan(index_type, "search")
        cgroup_props = json.dumps(search_settings.ImagesSettings.CGROUPS[index_type])

        if index_type == search_settings.INDEX_QUICK:
            basesearch_params = (analyze_task_rtyserver.BASESEARCH1_PARAMS, analyze_task_rtyserver.BASESEARCH2_PARAMS)
            analyze_task_type = analyze_task_rtyserver.ImagesAnalyzeRtyserverPerformance.type
        else:
            basesearch_params = (analyze_task.BASESEARCH1_PARAMS, analyze_task.BASESEARCH2_PARAMS)
            analyze_task_type = analyze_task.ImagesAnalyzeBasesearchPerformance.type

        sub_ctx = {
            dolbilka.DolbilkaExecutorMode.name: 'finger',
            dolbilka.DolbilkaMaximumSimultaneousRequests.name: 20,
            dolbilka.DolbilkaExecutorRequestsLimit.name: 500000,
            analyze_task.Plan1Parameter.name: plan_id,
            analyze_task.Plan2Parameter.name: plan_id,
            basesearch_params[0].Cgroup.name: cgroup_props,
            basesearch_params[1].Cgroup.name: cgroup_props,
        }

        if index_type == search_settings.INDEX_MAIN:
            sub_ctx.update({
                offline.WARMUP_TIME_KEY: 120,
                offline.SHUTDOWN_TIME_KEY: 2,
            })
        elif index_type == search_settings.INDEX_QUICK:
            sub_ctx.update({
                offline.WARMUP_TIME_KEY: 0,
                offline.SHUTDOWN_TIME_KEY: 0,
            })

        for params, age in zip(basesearch_params, ("old", "new")):
            sub_ctx.update({
                params.Binary.name: self._get_basesearch_executable(index_type),
                params.Config.name: self._get_basesearch_config(index_type),
                params.ArchiveModel.name: old_models_resource_id if age == "old" else new_models_resource_id
            })

            if index_type == search_settings.INDEX_QUICK:
                sub_ctx.update({
                    params.Database.name: self._get_basesearch_database(
                        index_type,
                        shard_name=(
                            rty_generate_plan_task.ImagesGenerateRtyserverRequests.get_shard_attrs_from_plan(plan_id)
                        ),
                    ),
                    params.StaticData.name: search_settings.ImagesSettings.get_rtyserver_static_data(),
                    params.ShardWriterConfig.name: search_settings.ImagesSettings.get_rtyserver_sw_config()
                })
            else:
                sub_ctx.update({
                    params.PatchRequestThreads.name: False,
                    params.PoliteMode.name: False,
                })

        sub_task = self.create_subtask(
            task_type=analyze_task_type,
            input_parameters=sub_ctx,
            description="{}, {}".format(self.descr, index_type),
            arch=sandboxapi.ARCH_LINUX,
            model='e5-2650'
        )

        return sub_task.id

    def __get_old_models_resource_id(self):
        return utils.get_and_check_last_released_resource_id(
            search_settings.ImagesSettings.models_resource(search_settings.INDEX_MAIN, search_settings.MODELS_PRODUCTION),
            arch=sandboxapi.ARCH_LINUX
        )


__Task__ = ImagesPriemkaBasesearchModels
