import logging

from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk import parameters

from sandbox.projects.common import apihelpers
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.common.mediasearch import release as media_release
from sandbox.projects.common.mediasearch import yasm_task
from sandbox.projects.images.models import ImagesBuildDynamicModels as build_task


class IndexTypeParameter(parameters.SandboxStringParameter):
    name = 'index_type'
    description = 'Index type'
    choices = [
        ('Main', search_settings.INDEX_MAIN),
        ('Middle', search_settings.INDEX_MIDDLE),
    ]
    default_value = search_settings.INDEX_MAIN


class ImagesReleaseDynamicModels(yasm_task.YasmTask,
                                 media_release.BaseNannyTask,
                                 media_release.BaseReleaseTask):
    """
        Build, test and release archive with experimental models
    """

    type = 'IMAGES_RELEASE_DYNAMIC_MODELS'

    input_parameters = (IndexTypeParameter,) + media_release.BaseReleaseTask.input_parameters

    @property
    def release_subject(self):
        return "images/exp-models-" + self.ctx[IndexTypeParameter.name] + "-{timestamp}"

    @property
    def release_comment(self):
        return "experimental images models ({})".format(self.ctx[IndexTypeParameter.name])

    @property
    def release_resources(self):
        return (self.__get_models_resource(),)

    @property
    def push_signal_name(self):
        return "release_{}_models".format(self.ctx[IndexTypeParameter.name])

    def _do_build(self):
        sub_ctx = {
            build_task.IndexTypeParameter.name: self.ctx[IndexTypeParameter.name],
            build_task.ModelsTypeParameter.name: search_settings.MODELS_EXPERIMENTAL,
        }
        sub_task = self.create_subtask(
            task_type=build_task.ImagesBuildDynamicModels.type,
            input_parameters=sub_ctx,
            description=self.descr,
        )
        return sub_task.id

    def _do_test(self, build_task_id):
        []

    def _skip_build(self):
        if not media_release.BaseReleaseTask._skip_build(self):
            return False

        svn_url = self.__get_models_url()
        svn_revision = str(svn.Arcadia.log(svn_url, "HEAD", 1, limit=1)[0]["revision"])

        released_resource = apihelpers.get_last_released_resource(self.__get_models_resource())
        if not released_resource:
            logging.info("No released resources. One needs to be built.")
            return False

        last_released_revision = str(released_resource.attributes.get("revision", "NONE"))
        logging.info("Last released revision: %s, recent SVN revision: %s", last_released_revision, svn_revision)
        return last_released_revision == svn_revision

    def _do_monitor(self):
        # TemporaryError is not compatible with 'Sequence run' schedulers
        self._yasm_notify(restart_on_failure=False)

    def __get_models_url(self):
        return search_settings.ImagesSettings.models_url(
            self.ctx[IndexTypeParameter.name],
            search_settings.MODELS_EXPERIMENTAL
        )

    def __get_models_resource(self, models_type=search_settings.MODELS_EXPERIMENTAL):
        return search_settings.ImagesSettings.models_resource(self.ctx[IndexTypeParameter.name], models_type)


__Task__ = ImagesReleaseDynamicModels
