import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths

from sandbox.projects import resource_types

from sandbox.projects.common.base_search_quality import response_saver
from sandbox.projects.common.dynamic_models import archiver as archiver_models
from sandbox.projects.common.dynamic_models import basesearch as basesearch_models
from sandbox.projects.common.dynamic_models import matrixnet as matrixnet_models
from sandbox.projects.common.search import requester_compat as search_requester
from sandbox.projects.images.basesearch import task as images_task
from sandbox.projects.images.resources import task as resources_task


RESPONSE_SAVER_PARAMS = response_saver.create_response_saver_params(required_queries=False)
_MODEL_PREFIX = "fml-mn-"


class MaxModelsNumberParameter(parameters.SandboxIntegerParameter):
    name = 'models_max_number'
    description = 'Max models number'
    default_value = 100


class ImagesTestBasesearchModels(resources_task.ImagesProductionResourcesTask, images_task.BaseImgsearchTask):
    """
        Test that basesearch recognize archive correctly
    """

    type = 'IMAGES_TEST_BASESEARCH_MODELS'
    client_tags = ctc.Tag.LINUX_PRECISE
    input_parameters = \
        (MaxModelsNumberParameter,) + \
        images_task.BaseImgsearchTask.basesearch_input_parameters + \
        RESPONSE_SAVER_PARAMS.params + \
        search_requester.create_params()

    execution_space = 150 * 1024  # 150 Gb

    def on_execute(self):
        queries_path = "queries.txt"
        responses_path = "responses.txt"

        archive_id = self.ctx[images_task.BASESEARCH_PARAMS.ArchiveModel.name]
        model_ids = self.__get_model_ids(archive_id)
        self.__create_plain_text_queries(model_ids, queries_path)

        basesearch = self._get_imgsearch()
        with basesearch:
            basesearch.use_component(lambda: search_requester.save_responses_old(self.ctx, queries_path, responses_path, basesearch))

        basesearch_models.verify_basesearch_responses("responses.txt", model_ids)
        self.set_info('{} formulas checked'.format(len(model_ids)))
        self.create_resource("{}, responses".format(self.descr), responses_path, resource_types.OTHER_RESOURCE)

    def _get_queries_parameter(self):
        class _FakeParameter:
            name = 'fake_parameter'
        return _FakeParameter

    def __get_model_ids(self, archive_id):
        models_dir = "models"
        paths.make_folder("models", delete_content=True)
        archiver_models.unpack(
            self.sync_resource(self._get_archiver_executable()),
            self.sync_resource(archive_id),
            models_dir
        )
        models_ids = list(matrixnet_models.get_models_id(self.sync_resource(self._get_mx_ops_executable()), [models_dir]))
        max_models = self.ctx[MaxModelsNumberParameter.name]
        if len(models_ids) > max_models:
            raise errors.SandboxTaskFailureError("Too many models ({} > {})".format(len(models_ids), max_models))
        return [v for k, v in models_ids]

    def __create_plain_text_queries(self, model_ids, queries_path):
        with open(queries_path, 'w') as output_file:
            for model_id in model_ids:
                model_num = model_id[len(_MODEL_PREFIX):] if model_id.startswith(_MODEL_PREFIX) else model_id
                output_file.write('?text=test&ms=proto&pron=exp_imagesformula_full.common_{}\n'.format(model_num))
        self.create_resource(self.descr, queries_path, resource_types.PLAIN_TEXT_QUERIES)


__Task__ = ImagesTestBasesearchModels
