import time
from collections import OrderedDict

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.common import utils
from sandbox.projects.images.bans import resources as images_bans_resources
from sandbox.projects.images.polish import resources as images_polish_resources


class ForceRunModeParameter(parameters.SandboxSelectParameter):
    FORCE_RUN_DISABLED = 0
    FORCE_PUBLISH_ONLY_MAN_VLA = FORCE_RUN_DISABLED + 1
    FORCE_ONLY_PUBLISH = FORCE_PUBLISH_ONLY_MAN_VLA + 1
    FULL_FORCE_RUN = FORCE_ONLY_PUBLISH + 1
    available_mode_names = OrderedDict()
    available_mode_names["None"] = FORCE_RUN_DISABLED
    available_mode_names["Publish only MAN & VLA"] = FORCE_PUBLISH_ONLY_MAN_VLA
    available_mode_names["Only publish (all locations)"] = FORCE_ONLY_PUBLISH
    available_mode_names["Full"] = FULL_FORCE_RUN

    name = "force_run_mode"
    description = "Force images manual boosting refresh"

    @classmethod
    def get_custom_parameters(self):
        return {"values": [(mode, mode) for mode in self.available_mode_names]}

    default_value = available_mode_names.keys()[0]
    required = True


class ServerParameter(parameters.SandboxStringParameter):
    name = "server"
    description = "YT server"
    default_value = "banach"
    required = True


class AttractivityTableParameter(parameters.SandboxStringParameter):
    name = "attractivity"
    description = "Attractivity data table"
    default_value = "//home/imgdev/saas/attractivity/attr_data"
    required = True


class SimilarityTableParameter(parameters.SandboxStringParameter):
    name = "similarity"
    description = "Similarity data table"
    default_value = "//home/imgdev/saas/attractivity/sim_cache"
    required = True


class VerificationTableParameter(parameters.SandboxStringParameter):
    name = "verification"
    description = "Verification data table"
    default_value = "//home/imgdev/saas/attractivity/sbs_data"
    required = True


class ShardDataTableParameter(parameters.SandboxStringParameter):
    name = "shard_data"
    description = "Saas shard data table"
    default_value = "//home/imgdev/saas/attractivity/shard_data"
    required = True


class BoilingShardDirParameter(parameters.SandboxStringParameter):
    name = "boiling_shard_dir"
    description = "Saas boiling shard dir"
    default_value = "//home/imgdev/saas/attractivity/boiling"
    required = True


class DelayFinalPublishParameter(parameters.SandboxIntegerParameter):
    name = "delay_saas_publish_shard_final"
    description = "Delay in hours"
    default_value = 8
    required = True


class ManualBoostingParameter(parameters.ResourceSelector):
    name = "manual_boosting_executable"
    description = "Manual boosting tool"
    resource_type = images_polish_resources.IMAGES_MANUAL_BOOSTING


class SaaSIndexerParameter(parameters.ResourceSelector):
    name = "saas_indexer_executable"
    description = "SaaS standalone indexer tool"
    resource_type = images_polish_resources.SAAS_STANDALONE_INDEXER


class LanguageDataParameter(parameters.SandboxArcadiaUrlParameter):
    name = "language_data_url"
    description = "SVN url of folder with language data"
    default_value = "arcadia:/arc/trunk/arcadia_tests_data/wizard/language"
    required = True


class SynnormDataParameter(parameters.SandboxArcadiaUrlParameter):
    name = "synnorm_data_url"
    description = "SVN url of folder with language data"
    default_value = "arcadia:/arc/trunk/arcadia_tests_data/wizard/synnorm"
    required = True


class ImagesManualBoostingRefresh(SandboxTask):
    """
        Builds and publishes SaaS shard for ImagesManualBoosting rule of Yandex.Images service
    """

    type = "IMAGES_MANUAL_BOOSTING_REFRESH"

    input_parameters = [
        ForceRunModeParameter,
        ServerParameter,
        AttractivityTableParameter,
        SimilarityTableParameter,
        VerificationTableParameter,
        ShardDataTableParameter,
        BoilingShardDirParameter,
        DelayFinalPublishParameter,
        LanguageDataParameter,
        SynnormDataParameter,
        ManualBoostingParameter,
        SaaSIndexerParameter,
    ]

    def _check_force_mode(self):
        return ForceRunModeParameter.available_mode_names[self.ctx[ForceRunModeParameter.name]]

    def _registered_data(self, resource):
        resource_last_state = utils.get_and_check_last_resource_with_attribute(resource)
        if not resource_last_state:
            return None

        resource_path = self.sync_resource(resource_last_state.id)
        with open(resource_path) as resource_ptr:
            return resource_ptr.read().strip()

    def _register_resource(self, path, resource):
        resource = self.create_resource(self.descr, path, resource)
        self.mark_resource_ready(resource.id, force_backup=True)

    def _register_resource_data(self, data, name, resource):
        data_path = self.abs_path(name)
        with open(data_path, "w") as data_ptr:
            data_ptr.write(data)
        self._register_resource(data_path, resource)

    def _manual_boosting(self, cmd, args=[]):
        manual_boosting_tool = self.sync_resource(self.ctx[ManualBoostingParameter.name])
        yt_token = self.get_vault_data("IMAGES-BAN", "yt_token")
        process.run_process([manual_boosting_tool, cmd, "--server", self.ctx[ServerParameter.name]] + args,
            environment={"YT_TOKEN": yt_token},
            outputs_to_one_file=False, log_prefix="manual_boosting.{}".format(cmd))

    def _prepare_shard_data(self, semidups_state):
        geobase_resource = utils.get_and_check_last_resource_with_attribute(images_bans_resources.IMAGES_GEOBASE5_SNAPSHOT)
        geobase_path = self.sync_resource(geobase_resource.id)

        language_path = self.abs_path("language")
        svn.Arcadia.export(self.ctx[LanguageDataParameter.name], language_path)

        synnorm_path = self.abs_path("synnorm")
        svn.Arcadia.export(self.ctx[SynnormDataParameter.name], synnorm_path)

        self.__manual_boosting("BuildRearrangeData", [
            "--semidups-prefix", "//home/images",
            "--semidups-state", semidups_state,
            "--wizard-language-dir", language_path,
            "--wizard-synnorm-dir", synnorm_path,
            "--geodata", geobase_path,
            "--job-memory-limit", "8589934592",
            "--attractivity-data-table", utils.get_or_default(self.ctx, AttractivityTableParameter),
            "--similarity-data-table", utils.get_or_default(self.ctx, SimilarityTableParameter),
            "--verification-data-table", utils.get_or_default(self.ctx, VerificationTableParameter),
            "--output-table", utils.get_or_default(self.ctx, ShardDataTableParameter)
        ])

    def _saas_indexer(self, ts, args=[], loc=""):
        saas_indexer_tool = self.sync_resource(self.ctx[SaaSIndexerParameter.name])
        yt_token = self.get_vault_data("IMAGES-BAN", "yt_token")

        rty_config_resource = utils.get_and_check_last_resource_with_attribute(images_polish_resources.SAAS_IMAGES_BUFFING_RTY_CONFIG)
        rty_config_path = self.sync_resource(rty_config_resource.id)
        search_map_resource = utils.get_and_check_last_resource_with_attribute(images_polish_resources.SAAS_IMAGES_BUFFING_SEARCH_MAP)
        search_map_path = self.sync_resource(search_map_resource.id)

        process.run_process([saas_indexer_tool, "yt", rty_config_path,
                             "--proxy", self.ctx[ServerParameter.name],
                             "--src", utils.get_or_default(self.ctx, ShardDataTableParameter),
                             "--dst-dir", utils.get_or_default(self.ctx, BoilingShardDirParameter),
                             "--service", "images-buffing",
                             "--searchmap", search_map_path,
                             "--processor", "message",
                             "--timestamp", ts,
                             "--verbose"
                             ] + args,
            environment={"YT_TOKEN": yt_token, "YT_USE_CLIENT_PROTOBUF": "0"},
            outputs_to_one_file=False, log_prefix="saas_indexer.{}.{}".format(ts, loc))

    def _saas_publish(self, ts, location):
        location_path = utils.get_or_default(self.ctx, BoilingShardDirParameter) + "/stable_kv/images-buffing/" + location
        self._saas_indexer(ts, ["--resume", "--publish", "--publish-path", location_path], location)

    def _check_publish_ts(self, ts, resource):
        saved_ts = self._registered_data(resource)
        if not saved_ts:
            return

        if int(saved_ts) < int(ts):
            return

        self._register_resource_data("0", "nul_timestamp", resource)

    def _make_copy_shard(self):
        import yt.wrapper as yt
        yt.config["proxy"]["url"] = self.ctx[ServerParameter.name]
        yt.config["token"] = self.get_vault_data("IMAGES-BAN", "yt_token")

        src = utils.get_or_default(self.ctx, ShardDataTableParameter)
        ts = self._registered_data(images_polish_resources.IMAGES_MANUAL_BOOSTING_SHARD_TS)
        state = self._registered_data(images_polish_resources.IMAGES_MANUAL_BOOSTING_VERSION)

        log_path = src + "_copy_log"
        MAX_COPIES_NUMBER = 5
        copies_path = [src + "_copy_" + str(i) for i in xrange(MAX_COPIES_NUMBER)]
        next_copy = {copies_path[i]: copies_path[(i + 1) % MAX_COPIES_NUMBER] for i in xrange(MAX_COPIES_NUMBER)}
        last_copy_path = None
        if yt.exists(log_path):
            log_data = yt.read_table(log_path, format=yt.JsonFormat())
            if log_data:
                last_copy_path = [x for x in log_data][-1]["copy_path"]

        next_copy_path = next_copy[last_copy_path] if last_copy_path else copies_path[0]
        yt.table_commands.copy_table(src, next_copy_path)
        log_info = {"copy_path": next_copy_path, "ts": ts, "version": state}
        yt.write_table(yt.TablePath(log_path, append=True), [log_info], format=yt.JsonFormat())

    def _check_version(self):
        semidups_state_path = self.abs_path("semidups_state")
        self.__manual_boosting("SelectSemidupsState", ["--index-prefix", "/home/images", "--index-state", "production", "--output-file", semidups_state_path])

        with open(semidups_state_path) as semidups_state_ptr:
            semidups_state = semidups_state_ptr.read().strip()

        if semidups_state and semidups_state == self._registered_data(images_polish_resources.IMAGES_MANUAL_BOOSTING_VERSION):
            if self._check_force_mode() != ForceRunModeParameter.FULL_FORCE_RUN:
                return

        self._prepare_shard_data(semidups_state)

        current_ts = str(int(time.time()))

        self._saas_indexer(current_ts)

        timestamp_path = self.abs_path("shard_timestamp")
        with open(timestamp_path, "w") as timestamp_ptr:
            print >>timestamp_ptr, current_ts

        self._register_resource_data(current_ts, "shard_timestamp", images_polish_resources.IMAGES_MANUAL_BOOSTING_SHARD_TS)
        self._check_publish_ts(current_ts, images_polish_resources.IMAGES_MANUAL_BOOSTING_SAS_TS)
        self._check_publish_ts(current_ts, images_polish_resources.IMAGES_MANUAL_BOOSTING_FINAL_TS)
        self._register_resource(semidups_state_path, images_polish_resources.IMAGES_MANUAL_BOOSTING_VERSION)

    def _check_publish_sas(self):
        saas_shard_ts = self._registered_data(images_polish_resources.IMAGES_MANUAL_BOOSTING_SHARD_TS)
        publish_sas_ts = self._registered_data(images_polish_resources.IMAGES_MANUAL_BOOSTING_SAS_TS)
        if not saas_shard_ts:
            return

        if publish_sas_ts and int(publish_sas_ts) >= int(saas_shard_ts):
            if self._check_force_mode() != ForceRunModeParameter.FORCE_ONLY_PUBLISH:
                return

        self._saas_publish(saas_shard_ts, "SAS")

        self._register_resource_data(str(int(time.time())), "sas_timestamp", images_polish_resources.IMAGES_MANUAL_BOOSTING_SAS_TS)

    def _check_publish_final(self):
        saas_shard_ts = self._registered_data(images_polish_resources.IMAGES_MANUAL_BOOSTING_SHARD_TS)
        publish_sas_ts = self._registered_data(images_polish_resources.IMAGES_MANUAL_BOOSTING_SAS_TS)
        publish_fin_ts = self._registered_data(images_polish_resources.IMAGES_MANUAL_BOOSTING_FINAL_TS)
        if not publish_sas_ts or not saas_shard_ts:
            return

        start_final_ts = int(publish_sas_ts) + self.ctx[DelayFinalPublishParameter.name] * 3600

        current_ts = int(time.time())
        if current_ts < start_final_ts:
            return

        if publish_fin_ts and int(publish_fin_ts) >= start_final_ts:
            if self._check_force_mode() != ForceRunModeParameter.FORCE_PUBLISH_ONLY_MAN_VLA:
                return

        self._saas_publish(saas_shard_ts, "MAN")
        self._saas_publish(saas_shard_ts, "VLA")

        self._register_resource_data(str(int(time.time())), "fin_timestamp", images_polish_resources.IMAGES_MANUAL_BOOSTING_FINAL_TS)
        self._make_copy_shard()

    def on_execute(self):
        self._check_version()
        self._check_publish_sas()
        self._check_publish_final()


__Task__ = ImagesManualBoostingRefresh
