import json
import os.path
import logging
import shutil
import tarfile
import time

import sandbox.common.types.task as ctt

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import task

from sandbox.projects.common import apihelpers
from sandbox.projects.common import decorators
from sandbox.projects.common import utils
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.images import ImagesBuildSearchBinary as build_task
from sandbox.projects.images import ImagesStoreShardmap as shardmap_task
from sandbox.projects.images.prism import resources as prism_resources


_SEMAPHORE_NAME = "IMAGES_PRISM_BASESEARCH_QUICK"
_SEMAPHORE_CAPACITY = 1

_READY_TIMEOUT = 10 * 60

_SCHEDULING_POLICY_ID = "scheduling_policy_id"
_TARGET_RESOURCE_ID = "target_resource_id"
_TARGET_SNAPSHOT_ID = "target_snapshot_id"
_STAGING_STATS_ID = "staging_stats"
_STAGING_PLOT_ID = "staging_plot_resource_id"


class IndexTypeParameter(parameters.SandboxStringParameter):
    name = 'index_type'
    description = 'Index type'
    choices = [
        ('Main', search_settings.INDEX_MAIN),
        ('Quick', search_settings.INDEX_QUICK),
        ('Cbirdaemon', search_settings.INDEX_CBIR_DAEMON),
        ('Naildaemon (quick)', search_settings.INDEX_THUMB_QUICK),
    ]
    default_value = search_settings.INDEX_MAIN


class BuildTaskParameter(parameters.TaskSelector):
    name = "build_task"
    description = 'build task'
    task_type = [build_task.ImagesBuildSearchBinary.type]
    required = True


class SwitchTimeoutParameter(parameters.SandboxIntegerParameter):
    name = "switching_timeout"
    description = "Switching timeout (min)"
    default_value = 50
    required = True


class WarmupTimeParameter(parameters.SandboxIntegerParameter):
    name = "warmup_time"
    description = "Warmup time (sec)"
    default_value = 120
    required = True


class RunningTimeParameter(parameters.SandboxIntegerParameter):
    name = "running_time"
    description = "Running time (sec)"
    default_value = 600
    required = True


class LimitsParameter(parameters.SandboxStringParameter):
    name = "limits"
    description = "Metric limits (leave empty for defaults)"
    required = False


class DeployModeParameter(parameters.SandboxStringParameter):
    DEPLOY_NONE = "none"
    DEPLOY_STAGING = "staging"
    DEPLOY_PRODUCTION = "production"

    name = 'deploy_mode'
    description = 'Deployment mode'
    default_value = DEPLOY_STAGING
    choices = [
        ('No deployment', DEPLOY_NONE),
        ('Deploy to staging', DEPLOY_STAGING),
        ('Deploy to production', DEPLOY_PRODUCTION)
    ]
    sub_fields = {
        DEPLOY_STAGING: [
            SwitchTimeoutParameter.name,
            WarmupTimeParameter.name,
            RunningTimeParameter.name,
            LimitsParameter.name,
        ],
        DEPLOY_PRODUCTION: [
            SwitchTimeoutParameter.name,
            WarmupTimeParameter.name,
            RunningTimeParameter.name,
            LimitsParameter.name,
        ],
    }


class VirtualenvParameter(parameters.ResourceSelector):
    name = 'virtualenv_resource_id'
    description = 'Prism virtualenv'
    resource_type = prism_resources.IMAGES_PRISM_VIRTUALENV
    required = False


class ImagesStagingBasesearch(task.SandboxTask):
    """
        Deploy basesearch to staging and monitor it for 10 minutes
    """

    type = 'IMAGES_STAGING_BASESEARCH'

    input_parameters = (
        IndexTypeParameter,
        BuildTaskParameter,
        DeployModeParameter,
        SwitchTimeoutParameter,
        WarmupTimeParameter,
        RunningTimeParameter,
        LimitsParameter,
        VirtualenvParameter,
    )

    @property
    def footer(self):
        items = []

        if _STAGING_STATS_ID in self.ctx:
            stats = self.ctx[_STAGING_STATS_ID]
            metrics = stats.keys()
            metrics.sort()
            items.append({
                "<h4>Difference between canary and production</h4>": {
                    "header": [
                        {"key": "metric", "title": "Metric"},
                        {"key": "p50", "title": "P50"},
                    ],
                    "body": {
                        "metric": metrics,
                        "p50": [stats[k]["p50"] for k in metrics]
                    }
                }
            })

        if _STAGING_PLOT_ID in self.ctx:
            picture_url = "//proxy.sandbox.yandex-team.ru/{}'".format(self.ctx[_STAGING_PLOT_ID])
            items.append({
                "<h4>Detailed</h4>": {
                    "<a href='{0}'><img style='width:100%' src='{0}'></a>".format(picture_url): ""
                }
            })

        if items:
            return [{"content": item} for item in items]
        else:
            return {"Calculating...": ""}

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        self.semaphores(ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(
                    name=self.__prism_semaphore_id(),
                    capacity=_SEMAPHORE_CAPACITY
                )
            ]
        ))

    def on_execute(self):
        index_type = self.ctx[IndexTypeParameter.name]
        deploy_mode = utils.get_or_default(self.ctx, DeployModeParameter)

        # clone resource to test
        if _TARGET_RESOURCE_ID not in self.ctx:
            binary_resource_id = self.__clone_resource(
                self.ctx[BuildTaskParameter.name],
                search_settings.ImagesSettings.basesearch_executable_resource(index_type, staging=False),
                search_settings.ImagesSettings.basesearch_executable_resource(index_type, staging=True)
            )
            config_resource_id = self.__clone_resource(
                self.ctx[BuildTaskParameter.name],
                search_settings.ImagesSettings.basesearch_config_resource(index_type, staging=False),
                search_settings.ImagesSettings.basesearch_config_resource(index_type, staging=True)
            )
            self.ctx[_TARGET_RESOURCE_ID] = [binary_resource_id, config_resource_id]

        # to boostrap deployment process we need
        # to have a first successful resource build
        if deploy_mode == DeployModeParameter.DEPLOY_NONE:
            return

        # wait until service is ready for changes (production specific)
        if (deploy_mode == DeployModeParameter.DEPLOY_PRODUCTION and
            _SCHEDULING_POLICY_ID not in self.ctx and _TARGET_SNAPSHOT_ID not in self.ctx):
            self.__nanny_wait_ready()

        # disable automation
        if _SCHEDULING_POLICY_ID not in self.ctx:
            self.ctx[_SCHEDULING_POLICY_ID] = self.__nanny_policy({"type": "NONE"})

        # sync with newdb (prism specific)
        if deploy_mode == DeployModeParameter.DEPLOY_STAGING and index_type == search_settings.INDEX_MAIN:
            self.__nanny_sync_newdb()

        # initiate deployment of specified resources
        if _TARGET_SNAPSHOT_ID not in self.ctx:
            result = self.__nanny_client().update_service_sandbox_file(
                self.__prism_service_id(),
                self.type,
                str(self.id),
                deploy=True,
                recipe=self.__prism_activate_recipe(),
                prepare_recipe=self.__prism_prepare_recipe(),
                skip_not_existing_resources=True,
                comment="Changed by task {}".format(self.id),
                deploy_comment="Deployed by task {}".format(self.id)
            )
            self.ctx[_TARGET_SNAPSHOT_ID] = result['set_target_state']['snapshot_id']

        try:
            # wait until target snapshot is ready
            self.__nanny_wait_active(self.ctx[_TARGET_SNAPSHOT_ID])

            # wait for service warmup
            warmup_time = self.ctx[WarmupTimeParameter.name]
            logging.info("Sleeping for {} sec".format(warmup_time))
            time.sleep(warmup_time)

            # gather monitoring data and calculate statistics
            self.__prism_run(
                "staging-monitor.py",
                "--instance-tags", self.__prism_service_tags(),
                "--chartsets-dir", self.__prism_chartsets_dir(),
                "--monitoring-time", str(self.ctx[RunningTimeParameter.name]),
                "--image-file", self.__prism_plot(),
                "--stats-file", self.__prism_stats(),
            )
            plot_resource = self.create_resource(
                self.descr,
                self.__prism_plot(),
                prism_resources.IMAGES_PRISM_PLOT
            )
            self.mark_resource_ready(plot_resource)
            self.ctx[_STAGING_PLOT_ID] = plot_resource.id
            with open(self.__prism_stats()) as f:
                self.ctx[_STAGING_STATS_ID] = json.load(f)

            # verify stats

            # Note: Different types of metrics has different meanings. For one metrics higher is better,
            # for other metrics lower is better. As a result we distinguish positive and negative thresholds.
            # If higher value is better, then threshold should be positive.
            # If lower value is better, then lower threshold should be negative.
            stats = self.ctx[_STAGING_STATS_ID]
            problems = []
            for metric, threshold in self.__prism_thresholds().iteritems():
                value = stats[metric]["p50"]
                if threshold > 0 and value > threshold or threshold < 0 and value < threshold:
                    problems.append((metric, value, threshold))
            if problems:
                text = "Some of metrics exceed their thresholds:\n"
                text += "\n".join([
                    "{}: {} > {}".format(metric, actual_value, threshold)
                    for metric, actual_value, threshold in problems
                ])
                raise errors.SandboxTaskFailureError(text)
        finally:
            self.__nanny_policy(self.ctx[_SCHEDULING_POLICY_ID])

    def __prism_semaphore_id(self):
        # TODO: replace with service_id after switching from old prism tasks
        index_type = self.ctx[IndexTypeParameter.name]
        if index_type == search_settings.INDEX_MAIN:
            return "IMAGES_PRISM_BASESEARCH_MAIN"
        elif index_type == search_settings.INDEX_QUICK:
            return "IMAGES_PRISM_BASESEARCH_QUICK"
        elif index_type == search_settings.INDEX_THUMB_QUICK:
            return "IMAGES_PRISM_NAILDAEMON"
        elif index_type == search_settings.INDEX_CBIR_DAEMON:
            return "IMAGES_PRISM_CBIRDAEMON"
        else:
            raise errors.SandboxTaskFailureError("Unsupported index_type {}".format(index_type))

    def __prism_service_id(self):
        index_type = self.ctx[IndexTypeParameter.name]
        staging = utils.get_or_default(self.ctx, DeployModeParameter) == DeployModeParameter.DEPLOY_STAGING
        return search_settings.ImagesSettings.service_id(index_type, staging=staging)

    def __prism_service_tags(self):
        index_type = self.ctx[IndexTypeParameter.name]
        staging = utils.get_or_default(self.ctx, DeployModeParameter) == DeployModeParameter.DEPLOY_STAGING
        return search_settings.ImagesSettings.service_tags(index_type, staging=staging)

    def __prism_prepare_recipe(self):
        index_type = self.ctx[IndexTypeParameter.name]
        deploy_mode = utils.get_or_default(self.ctx, DeployModeParameter)
        if deploy_mode == DeployModeParameter.DEPLOY_PRODUCTION:
            if index_type == search_settings.INDEX_CBIR_DAEMON:
                return None
        return "fast_prepare"

    def __prism_activate_recipe(self):
        index_type = self.ctx[IndexTypeParameter.name]
        deploy_mode = utils.get_or_default(self.ctx, DeployModeParameter)
        if deploy_mode == DeployModeParameter.DEPLOY_PRODUCTION:
            if index_type == search_settings.INDEX_CBIR_DAEMON:
                return "activate"
        return "fast_activate"

    def __prism_plot(self):
        return "monitoring.png"

    def __prism_stats(self):
        return "stats.json"

    def __prism_thresholds(self):
        value = self.ctx[LimitsParameter.name]
        if value:
            return json.loads(value)
        else:
            return {}

    def __prism_chartsets_dir(self):
        virtualenv_dir = self.__prism_virtualenv_dir()
        return os.path.join(virtualenv_dir, "prism", "charts")

    def __prism_run(self, tool, *args):
        virtualenv_dir = self.__prism_virtualenv_dir()
        cmd = (os.path.join(virtualenv_dir, "bin", "python"), os.path.join(virtualenv_dir, "prism", tool)) + args
        process.run_process(cmd, log_prefix=tool)

    @decorators.memoize
    def __prism_virtualenv_dir(self):
        virtualenv_resource_id = utils.get_or_default(self.ctx, VirtualenvParameter)
        if not virtualenv_resource_id:
            virtualenv_resource_id = utils.get_and_check_last_released_resource_id(prism_resources.IMAGES_PRISM_VIRTUALENV)
        virtualenv_archive = self.sync_resource(virtualenv_resource_id)
        virtualenv_path = self.abs_path("venv")
        paths.make_folder(virtualenv_path, delete_content=True)
        tarfile.open(virtualenv_archive, "r:*").extractall(virtualenv_path)
        return virtualenv_path

    def __clone_resource(self, task_id, source_resource_type, target_resource_type):
        task_resources = apihelpers.list_task_resources(task_id, source_resource_type)
        if not task_resources:
            raise errors.SandboxTaskFailureError(
                "Failed to find resource {} in task {}".format(source_resource_type, task_id)
            )
        source_path = self.sync_resource(task_resources[0].id)
        target_path = os.path.basename(source_path)
        shutil.copyfile(source_path, target_path)
        target_resource = self.create_resource(
            self.descr,
            target_path,
            target_resource_type,
            arch=task_resources[0].arch
        )
        self.mark_resource_ready(target_resource)
        return target_resource.id

    def __nanny_policy(self, new_policy):
        attrs = self.__nanny_client().get_service_info_attrs(self.__prism_service_id())
        snapshot_id = attrs['_id']
        orig_policy = attrs["content"]["scheduling_policy"]
        logging.info("attrs={}, orig_policy={}, new_policy={}".format(attrs, orig_policy, new_policy))
        attrs["content"]["scheduling_policy"] = new_policy
        self.__nanny_client().set_service_info_attrs(self.__prism_service_id(), {
            "content": attrs['content'],
            "snapshot_id": snapshot_id,
            "comment": "Task {}".format(self.id),
        })
        return orig_policy

    def __nanny_wait_ready(self):
        waitfor = time.time() + _READY_TIMEOUT
        while time.time() < waitfor:
            current_state = self.__nanny_client().get_service_current_state(self.__prism_service_id())
            if (current_state['content']['summary']['value'] == 'ONLINE' and
                current_state['content']['is_paused']['value'] == False):
                return
            time.sleep(60)
        raise errors.SandboxTaskFailureError("Failed to wait until service ready to change (timeout={}s)".format(_READY_TIMEOUT))

    def __nanny_wait_active(self, target_snapshot_id):
        switch_timeout = self.ctx[SwitchTimeoutParameter.name] * 60
        waitfor = time.time() + switch_timeout
        while time.time() < waitfor:
            current_state = self.__nanny_client().get_service_current_state(self.__prism_service_id())
            if current_state['content']['summary']['value'] not in ('ONLINE', 'UPDATING', 'PREPARING'):
                raise Exception('Service is offline')
            for snapshot in current_state['content']['active_snapshots']:
                if snapshot['state'] == 'ACTIVE' and snapshot['snapshot_id'] == target_snapshot_id:
                    return
            time.sleep(60)
        raise errors.SandboxTaskFailureError("Failed to switch service to new state (timeout={}s)".format(switch_timeout))

    def __nanny_sync_newdb(self):
        prod_service_id = self.__prism_service_id()
        newdb_service_id = "prism_newdb_imgsbase_man"
        shardmap_task_type = shardmap_task.ImagesStoreShardmap.type

        logging.info("Trying to synchronize shardmap between {} and {}".format(
            prod_service_id, newdb_service_id
        ))

        prod_shardmap_task_id = self.__nanny_shardmap(prod_service_id)
        newdb_shardmap_task_id = self.__nanny_shardmap(newdb_service_id)
        if prod_shardmap_task_id == newdb_shardmap_task_id:
            return

        self.__nanny_client().update_service_sandbox_bsc_shardmap(
            prod_service_id,
            shardmap_task_type,
            newdb_shardmap_task_id
        )

    @decorators.memoize
    def __nanny_shardmap(self, service_id):
        return self.__nanny_resource(service_id)['sandbox_bsc_shard']['sandbox_shardmap']['task_id']

    @decorators.retries(max_tries=3, delay=10)
    def __nanny_resource(self, service_id):
        nanny_client = self.__nanny_client()

        # search for active snapshot id
        # TODO: simplify after SWAT-2375
        current_state = nanny_client.get_service_current_state(service_id)
        if current_state['content']['summary']['value'] not in ('ONLINE', 'UPDATING', 'PREPARING'):
            raise errors.SandboxTaskFailureError('Service is offline')
        for snapshot in current_state['content']['active_snapshots']:
            if snapshot['state'] in ('ACTIVATING', 'GENERATING', 'PREPARING'):
                active_snapshot_id = current_state['content']['rollback_snapshot']['snapshot_id']
                break
            elif snapshot['state'] in ('ACTIVE'):
                active_snapshot_id = snapshot['snapshot_id']
                break
        else:
            raise errors.SandboxTaskFailureError('Failed to find active snapshot')

        # retrieve information about shardmap
        runtime_attrs = nanny_client.get_history_runtime_attrs(active_snapshot_id)
        return runtime_attrs['content']['resources']

    @decorators.memoize
    def __nanny_client(self):
        deploy_mode = utils.get_or_default(self.ctx, DeployModeParameter)
        if deploy_mode == DeployModeParameter.DEPLOY_STAGING:
            sandbox_group = "IMAGES-SANDBOX"
        elif deploy_mode == DeployModeParameter.DEPLOY_PRODUCTION:
            sandbox_group = "MEDIA_DEPLOY"
        else:
            raise errors.SandboxTaskFailureError("Unsupported mode for deployment")

        return nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data(sandbox_group, 'nanny-oauth-token')
        )


__Task__ = ImagesStagingBasesearch
