import json
import logging
import os.path
from sandbox import sdk2
import sandbox.sdk2.helpers as sdk2_helpers
import shutil
import tarfile
import time

from sandbox.common import errors
import sandbox.common.types.client as ctc
import sandbox.common.types.task as task_types
import sandbox.sdk2.paths as sdk_paths

from sandbox.projects import DeployNannyDashboard as deploy_nanny_dashboard
from sandbox.projects import MediaTaskMonitoring as media_task_monitoring
from sandbox.projects.common import decorators
from sandbox.projects.common.search import settings as search_settings
from sandbox.projects.common.nanny import nanny
from sandbox.projects.images.deployment import zookeeper
from sandbox.projects.images.prism import resources as prism_resources


_WAIT_STATUSES = task_types.Status.Group.FINISH + task_types.Status.Group.BREAK

_MONITORING_CHAT = "-1001088652476"  # Chat to notify about problems
_MONINORING_EMAIL = ""  # User to notify about problems
_MONITORING_ALERT_TIME = 1 * 60 * 60  # Notify about problems after specified time
_MONITORING_SLEEP_TIME = 2 * 60  # Time to sleep between checks in deployment and monitoring tasks

_DEPLOY_NONE = "none"
_DEPLOY_STAGING = "staging"
_DEPLOY_PRODUCTION = "production"


class ImagesStagingBasesearch2(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
        Deploy basesearch to staging and monitor it for 10 minutes

        New version with experimental deployment schema
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC

    class Parameters(sdk2.Task.Parameters):
        build_task = sdk2.parameters.Task("Build task", required=True)
        index_type = sdk2.parameters.String(
            "Index type",
            default=search_settings.INDEX_MAIN,
            choices=[
                ('Main', search_settings.INDEX_MAIN),
                ('Quick', search_settings.INDEX_QUICK),
                ('Cbirdaemon', search_settings.INDEX_CBIR_DAEMON),
                ('Naildaemon (quick)', search_settings.INDEX_THUMB_QUICK),
            ],
            required=True
        )
        deploy_mode = sdk2.parameters.String(
            "Deployment mode",
            default=_DEPLOY_STAGING,
            choices=[
                ('No deployment (build only)', _DEPLOY_NONE),
                ('Deploy to staging', _DEPLOY_STAGING),
                ('Deploy to production', _DEPLOY_PRODUCTION)
            ],
            required=True
        )
        zookeeper_force = sdk2.parameters.Bool(
            "Force to obtain zookeeper lock (experts only)",
            default=False
        )
        validation_warmup_time = sdk2.parameters.Integer(
            "Warmup time (sec)",
            default=120,
            required=True
        )
        validation_running_time = sdk2.parameters.Integer(
            "Running time (sec)",
            default=600,
            required=True
        )
        validation_limits = sdk2.parameters.String(
            "Metric limits (leave empty for defaults)"
        )
        validation_virtualenv = sdk2.parameters.Resource(
            "Prism virtualenv",
            resource_type=prism_resources.IMAGES_PRISM_VIRTUALENV
        )

    @property
    def footer(self):
        items = []

        stats = self.Context.stats
        if stats:
            metrics = stats.keys()
            metrics.sort()
            items.append({
                "<h4>Difference between canary and production</h4>": {
                    "header": [
                        {"key": "metric", "title": "Metric"},
                        {"key": "p50", "title": "P50"},
                    ],
                    "body": {
                        "metric": metrics,
                        "p50": [stats[k]["p50"] for k in metrics]
                    }
                }
            })

        plot_resource_id = self.Context.plot_resource_id
        if plot_resource_id:
            picture_url = "//proxy.sandbox.yandex-team.ru/{}'".format(plot_resource_id)
            items.append({
                "<h4>Detailed</h4>": {
                    "<a href='{0}'><img style='width:100%' src='{0}'></a>".format(picture_url): ""
                    }
            })

        if items:
            return [{"content": item} for item in items]
        else:
            return {"Calculating...": ""}

    def on_finish(self, prev_status, status):
        if self.Parameters.deploy_mode != _DEPLOY_NONE:
            zookeeper.zk_delete(self.__dashboard_lock())

    def on_execute(self):
        # Code below is an emulation of 'fail_on_any_error' with a more simple and correct behaviour
        try:
            self.__on_execute()
        except (errors.TaskFailure, sdk2.WaitTask):
            raise
        except Exception as e:
            raise errors.TaskFailure("Unexpected exception: {}".format(e))

    def __on_execute(self):
        if self.Parameters.deploy_mode == _DEPLOY_NONE:
            index_type = self.Parameters.index_type
            self.__clone_resource(
                search_settings.ImagesSettings.basesearch_executable_resource(index_type, staging=False),
                search_settings.ImagesSettings.basesearch_executable_resource(index_type, staging=True)
            )
            self.__clone_resource(
                search_settings.ImagesSettings.basesearch_config_resource(index_type, staging=False),
                search_settings.ImagesSettings.basesearch_config_resource(index_type, staging=True)
            )
            return

        # Build & release
        with self.memoize_stage.builder(commit_on_entrance=False):
            build_task = ImagesStagingBasesearch2(
                self,
                description="{}, builder".format(self.Parameters.description),
                build_task=self.Parameters.build_task,
                index_type=self.Parameters.index_type,
                deploy_mode=_DEPLOY_NONE,
            ).enqueue()
            self.Context.build_task_id = build_task.id
            raise sdk2.WaitTask([build_task], _WAIT_STATUSES)

        build_task = sdk2.Task[self.Context.build_task_id]
        if build_task.status == task_types.Status.SUCCESS:
            self.server.release(
                task_id=build_task.id,
                type=self.__dashboard_release_type(),
                subject="Automatic release for staging (task #{})".format(self.id)
            )
            raise sdk2.WaitTask([build_task], _WAIT_STATUSES)
        elif build_task.status != task_types.Status.RELEASED:
            raise errors.TaskFailure("Build task failed")

        # Deploy
        with self.memoize_stage.deployer(commit_on_entrance=False):
            zookeeper_path = self.__dashboard_lock()
            if self.Parameters.zookeeper_force:
                zookeeper.zk_delete(zookeeper_path)
            zookeeper.zk_create_exclusive(zookeeper_path, value=str(self.id))

            self.Context.deployment_task_id = self.__run_deployment_task(self.Context.build_task_id).id
            self.Context.monitoring_task_id = self.__run_monitoring_task(self.Context.deployment_task_id).id

        self.__wait_deployment_tasks(self.Context.deployment_task_id, self.Context.monitoring_task_id)

        # Monitor
        warmup_time = self.Parameters.validation_warmup_time
        logging.info("Sleeping for {} sec".format(warmup_time))
        time.sleep(warmup_time)

        self.__prism_run(
            "staging-monitor.py",
            "--instance-tags", self.__prism_service_tags(),
            "--chartsets-dir", self.__prism_chartsets_dir(),
            "--monitoring-time", str(self.Parameters.validation_warmup_time),
            "--image-file", self.__prism_plot(),
            "--stats-file", self.__prism_stats(),
        )

        plot_resource = prism_resources.IMAGES_PRISM_PLOT(
            self,
            self.Parameters.description,
            self.__prism_plot()
        )
        sdk2.ResourceData(plot_resource).ready()
        self.Context.plot_resource_id = plot_resource.id

        with open(self.__prism_stats()) as f:
            self.Context.stats = json.load(f)

        # Validate
        # Note: Different types of metrics has different meanings. For one metrics higher is better,
        # for other metrics lower is better. As a result we distinguish positive and negative thresholds.
        # If higher value is better, then threshold should be positive.
        # If lower value is better, then lower threshold should be negative.
        stats = self.Context.stats
        problems = []
        for metric, threshold in self.__prism_thresholds().iteritems():
            value = stats[metric]["p50"]
            if threshold > 0 and value > threshold or threshold < 0 and value < threshold:
                problems.append((metric, value, threshold))
        if problems:
            text = "Some of metrics exceed their thresholds:\n"
            text += "\n".join([
                "{}: {} > {}".format(metric, actual_value, threshold)
                for metric, actual_value, threshold in problems
            ])
            raise errors.TaskFailure(text)

    def __clone_resource(self, source_resource_type, target_resource_type):
        source_resource_obj = sdk2.Resource.find(source_resource_type, task=self.Parameters.build_task).first()
        if not source_resource_obj:
            raise errors.TaskFailure("Failed to find resource {} in build task".format(
                source_resource_type
            ))
        source_resource_path = str(sdk2.ResourceData(source_resource_obj).path)
        target_resource_path = os.path.basename(source_resource_path)
        shutil.copyfile(source_resource_path, target_resource_path)
        target_resource_type(
            self,
            self.Parameters.description,
            target_resource_path,
        )

    def __run_deployment_task(self, build_task_id):
        sub_ctx = {
            deploy_nanny_dashboard.ReleaseTask.name: build_task_id,
            deploy_nanny_dashboard.NannyDashboardName.name: self.__dashboard_name(),
            deploy_nanny_dashboard.NannyDashboardRecipeName.name: self.__dashboard_recipe(),
            deploy_nanny_dashboard.VaultOwner.name: self.__dashboard_vault_owner(),
            deploy_nanny_dashboard.SandboxReleaseType.name: self.__dashboard_release_type(),
            deploy_nanny_dashboard.WaitDeployment.name: _MONITORING_SLEEP_TIME,
            deploy_nanny_dashboard.NannyWaitDeployParameter.name: True,
            deploy_nanny_dashboard.GetServicesFromRecipe.name: True,
        }
        sub_task_class = sdk2.Task[deploy_nanny_dashboard.DeployNannyDashboard.type]
        return sub_task_class(self, **sub_ctx).enqueue()

    def __run_monitoring_task(self, deployment_task_id):
        sub_ctx = {
            media_task_monitoring.MonitoringTaskId.name: deployment_task_id,
            media_task_monitoring.TelegramChatId.name: _MONITORING_CHAT,
            media_task_monitoring.Email.name: _MONINORING_EMAIL,
            media_task_monitoring.MonitoringTime.name: _MONITORING_ALERT_TIME,
            media_task_monitoring.MonitoringSleep.name: _MONITORING_SLEEP_TIME,
            media_task_monitoring.VaultOwner.name: self.__telegram_vault_owner(),
        }
        sub_task_class = sdk2.Task[media_task_monitoring.MediaTaskMonitoring.type]
        return sub_task_class(self, **sub_ctx).enqueue()

    def __wait_deployment_tasks(self, deployment_task_id, monitoring_task_id):
        wait_list, bad_list = [], []
        for task_id in (deployment_task_id, monitoring_task_id):
            task = sdk2.Task[task_id]
            if task.status not in _WAIT_STATUSES:
                wait_list.append(task)
            elif task.status != task_types.Status.SUCCESS:
                bad_list.append(task)

        logging.info("wait_deployment_tasks: wait_list={}, bad_list={}".format(wait_list, bad_list))

        if bad_list:
            raise errors.TaskFailure("Deployment task failed")

        if wait_list:
            raise sdk2.WaitTask(wait_list, _WAIT_STATUSES, wait_all=False)

    def __dashboard_name(self):
        deploy_mode = self.Parameters.deploy_mode
        if deploy_mode == _DEPLOY_STAGING:
            return "images_prism"
        elif deploy_mode == _DEPLOY_PRODUCTION:
            return "images_runtime"
        else:
            raise errors.TaskFailure("Unsupported deploy_mode: {}".format(deploy_mode))

    def __dashboard_recipe(self):
        index_type = self.Parameters.index_type
        if index_type == search_settings.INDEX_CBIR_DAEMON:
            return "cbrd_deploy_canary"
        else:
            raise errors.TaskFailure("Unsupported index_type: {}".format(index_type))

    def __telegram_vault_owner(self):
        return "MEDIA_DEPLOY"

    def __dashboard_vault_owner(self):
        deploy_mode = self.Parameters.deploy_mode

        if deploy_mode == _DEPLOY_STAGING:
            return "IMAGES-SANDBOX"
        elif deploy_mode == _DEPLOY_PRODUCTION:
            return "MEDIA_DEPLOY"
        else:
            raise errors.TaskFailure("Unsupported deploy_mode: {}".format(deploy_mode))

    def __dashboard_release_type(self):
        deploy_mode = self.Parameters.deploy_mode

        if deploy_mode == _DEPLOY_STAGING:
            return task_types.ReleaseStatus.PRESTABLE
        elif deploy_mode == _DEPLOY_PRODUCTION:
            return task_types.ReleaseStatus.STABLE
        else:
            raise errors.TaskFailure("Unsupported deploy_mode: {}".format(deploy_mode))

    def __dashboard_lock(self):
        index_type = self.Parameters.index_type
        staging = self.Parameters.deploy_mode == _DEPLOY_STAGING
        return search_settings.ImagesSettings.zookeeper_path(index_type, staging=staging)

    def __prism_service_tags(self):
        index_type = self.Parameters.index_type
        staging = self.Parameters.deploy_mode == _DEPLOY_STAGING

        return search_settings.ImagesSettings.service_tags(index_type, staging=staging)

    def __prism_plot(self):
        return "monitoring.png"

    def __prism_stats(self):
        return "stats.json"

    def __prism_chartsets_dir(self):
        virtualenv_dir = self.__prism_virtualenv_dir()
        return os.path.join(virtualenv_dir, "prism", "charts")

    def __prism_thresholds(self):
        value = self.Parameters.validation_limits
        if value:
            return json.loads(value)
        else:
            return {}

    def __prism_run(self, tool, *args):
        virtualenv_dir = self.__prism_virtualenv_dir()
        cmd = (os.path.join(virtualenv_dir, "bin", "python"), os.path.join(virtualenv_dir, "prism", tool)) + args

        with sdk2_helpers.ProcessLog(self, logger=logging.getLogger("prism")) as pl:
            sdk2_helpers.subprocess.check_call(cmd, stdout=pl.stdout, stderr=sdk2_helpers.subprocess.STDOUT)

    @decorators.memoize
    def __prism_virtualenv_dir(self):
        virtualenv_resource = self.Parameters.validation_virtualenv

        if not virtualenv_resource:
            virtualenv_resource = prism_resources.IMAGES_PRISM_VIRTUALENV.find(
                attrs={"released": task_types.ReleaseStatus.STABLE}
            ).first()

        if not virtualenv_resource:
            raise errors.TaskFailure("Failed to find virtualenv resource")

        virtualenv_archive = str(sdk2.ResourceData(virtualenv_resource).path)
        virtualenv_path = "venv"
        sdk_paths.make_folder(virtualenv_path, delete_content=True)
        tarfile.open(virtualenv_archive, "r:*").extractall(virtualenv_path)
        return virtualenv_path
