import argparse
import glob
import json
import logging
import numpy
import re
import time

import matplotlib
matplotlib.use("Agg")
import matplotlib.pyplot as plt


HOST = "ASEARCH"
INSTANCE_TYPE_RE = re.compile(r"(?:^|;)itype=([^;]+)(?:$|;)")


def _normalized_x(signals_data):
    x = [k for k, _ in signals_data]
    x_min = x[0]
    return [v - x_min for v in x]


def _signal_y(instance_tags, signal, points, default_value=None):
    tags = "{};{}".format(instance_tags, signal["tag"])
    return [point.get(tags, {}).get(signal["name"], default_value) for _, point in points]


def _load_chartsets(instance_tags, chartsets_dir):
    m = INSTANCE_TYPE_RE.search(instance_tags)
    if not m:
        raise Exception("Failed to find itype in instance tags")
    instance_type = m.group(1)

    for chartset_path in glob.glob("{}/{}.*.conf".format(chartsets_dir, instance_type)):
        with open(chartset_path) as chartset_file:
            try:
                yield json.load(chartset_file)
            except Exception, e:
                raise Exception("Failed to load: {}: {}".format(chartset_path, e))


def query_signals(instance_tags, chartsets_dir, monitoring_time):
    """
        Query all signals we need for chartsets from specified directory

        Realtime API used
    """

    import yasmapi

    # We need at least one active signal to avoid stuck in iterator (GOLOVAN-4568)
    signals = {instance_tags: ["const(1)"]}
    for chartset_data in _load_chartsets(instance_tags, chartsets_dir):
        for signal in chartset_data["signals"]:
            tags = "{};{}".format(instance_tags, signal["tag"])
            signals.setdefault(tags, []).append(signal["name"])

    end_time = time.time() + monitoring_time
    points = []
    for point in yasmapi.RtGolovanRequest({HOST: signals}):
        cur_time = point.ts
        if cur_time >= end_time:
            break
        points.append((point.ts, point.values[HOST]))
    return points


def draw_signals(instance_tags, chartsets_dir, points, image_file):
    """Render all chartsets from specified directory"""

    plt.figure(figsize=(18, 21))

    plot_count = sum(1 for _ in _load_chartsets(instance_tags, chartsets_dir))
    logging.info("Total chartset count - {}".format(plot_count))
    for plot_num, chartset_data in enumerate(_load_chartsets(instance_tags, chartsets_dir)):
        plt.subplot(plot_count, 1, plot_num + 1)
        plt.title(chartset_data["title"])

        x = _normalized_x(points)
        for signal in chartset_data["signals"]:
            y = _signal_y(instance_tags, signal, points)
            plt.plot(x, y, color=signal["color"], label=signal["title"])
            plt.gca().legend(fontsize="x-small")

    plt.tight_layout()
    plt.savefig(image_file)


# TODO: Use pandas and skip values with n/a
def draw_stats(instance_tags, chartsets_dir, points, stats_file):
    """Store main statistics in json file"""

    def _find_signal(chartset_data, title):
        for signal in chartset_data["signals"]:
            if signal["title"] == title:
                return signal
        raise Exception("Failed to find signal named '{}'".format(title))

    stats = {}
    for chartset_data in _load_chartsets(instance_tags, chartsets_dir):
        current_signal = _find_signal(chartset_data, "current")
        canary_signal = _find_signal(chartset_data, "canary")

        current_y = numpy.array(_signal_y(instance_tags, current_signal, points, 0))
        canary_y = numpy.array(_signal_y(instance_tags, canary_signal, points, 0))
        y = current_y - canary_y

        stats[chartset_data["title"]] = {
            "min": y.min(),
            "max": y.max(),
            "avg": y.mean(),
            "p50": numpy.median(y)
        }

    with open(stats_file, "w") as f:
        json.dump(stats, f)


def main():
    parser = argparse.ArgumentParser(description="Fetch monitoring signals")

    parser.add_argument("--instance-tags", type=str, required=True, help="instance tags")
    parser.add_argument("--chartsets-dir", type=str, required=True, help="directory with chartsets")
    parser.add_argument("--monitoring-time", type=int, required=True, help="monitoring period back from now")
    parser.add_argument("--image-file", type=str, required=True, help="file with plots")
    parser.add_argument("--stats-file", type=str, required=True, help="file with stats")

    args = parser.parse_args()

    points = query_signals(
        instance_tags=args.instance_tags,
        chartsets_dir=args.chartsets_dir,
        monitoring_time=args.monitoring_time,
    )
    draw_signals(
        instance_tags=args.instance_tags,
        chartsets_dir=args.chartsets_dir,
        points=points,
        image_file=args.image_file
    )
    draw_stats(
        instance_tags=args.instance_tags,
        chartsets_dir=args.chartsets_dir,
        points=points,
        stats_file=args.stats_file
    )


if __name__ == "__main__":
    main()
