import urllib

from sandbox.projects.common.search.settings import ImagesSettings

from sandbox.projects.common.pumpkin import serp_task
from sandbox.projects.common.mediasearch import yasm_task
from sandbox.projects.images.pumpkin import ImagesTestPumpkinSerp as test_task
from sandbox.projects.images.pumpkin import resources as images_pumpkin_resources

from sandbox.sandboxsdk.parameters import SandboxStringParameter


_PUMPKIN_CGI_PARAMS = "exp_flags=pumpkin&no-tests=1&test-mode=1"
_NOTFOUND_CGI_PARAMS = "{0}&site=fake".format(_PUMPKIN_CGI_PARAMS)
_EXECUTED_KEY = "_task_executed"


_PLATFORM_DESKTOP = "desktop"
_PLATFORM_TOUCH = "touch"


class PlatformParameter(SandboxStringParameter):
    name = 'platform'
    description = 'Platform'
    choices = [(_PLATFORM_DESKTOP, _PLATFORM_DESKTOP), (_PLATFORM_TOUCH, _PLATFORM_TOUCH)]
    default_value = _PLATFORM_DESKTOP
    required = True


class ImagesBuildPumpkinSerp(yasm_task.YasmTask, serp_task.BaseBuildPumpkinSerpTask):
    type = "IMAGES_BUILD_PUMPKIN_SERP"

    input_parameters = (PlatformParameter,) + serp_task.BaseBuildPumpkinSerpTask.input_parameters

    INDEX_RESOURCE_TYPE = images_pumpkin_resources.IMAGES_PUMPKIN_INDEX
    ARCHIVE_RESOURCE_TYPE = images_pumpkin_resources.IMAGES_PUMPKIN_INDEX_ARCHIVE

    def on_execute(self):
        # yasm_notify can raise TemporaryError, so we need to avoid a double work
        if _EXECUTED_KEY not in self.ctx:
            serp_task.BaseBuildPumpkinSerpTask.on_execute(self)
            self.ctx[_EXECUTED_KEY] = 1

        domain = self.ctx[serp_task.DomainParameter.name]
        self._yasm_notify(push_signal_name="build_pumpkin_serp_{}".format(domain))

    def _make_production_attributes(self, domain):
        return ImagesSettings.pumpkin_resource_attributes(domain)

    def _make_serp_url(self, domain, text):
        if self.ctx[PlatformParameter.name] == _PLATFORM_TOUCH:
            cgi_params = '{0}&noredirect=1'.format(_PUMPKIN_CGI_PARAMS)
        else:
            cgi_params = _PUMPKIN_CGI_PARAMS

        return "{0}/search?text={1}&{2}".format(self._make_url(domain), urllib.quote(text), cgi_params)

    def _make_main_page_url(self, domain):
        if self.ctx[PlatformParameter.name] == _PLATFORM_TOUCH:
            cgi_params = '{0}&exp_flags=images_collections_proxy%3D0&noredirect=1'.format(_PUMPKIN_CGI_PARAMS)
        else:
            cgi_params = _PUMPKIN_CGI_PARAMS

        return "{0}?{1}".format(self._make_url(domain), cgi_params)

    def _make_notfound_page_url(self, domain):
        if self.ctx[PlatformParameter.name] == _PLATFORM_TOUCH:
            cgi_params = '{0}&noredirect=1'.format(_NOTFOUND_CGI_PARAMS)
        else:
            cgi_params = _NOTFOUND_CGI_PARAMS

        return "{0}/search?{1}".format(self._make_url(domain), cgi_params)

    def _make_url(self, domain):
        url = serp_task.BaseBuildPumpkinSerpTask._make_url(self, domain)
        if url is not None:
            return url

        if domain == serp_task.RU_DOMAIN:
            url = "https://yandex.ru/images"
        else:
            url = "https://yandex.com.tr/gorsel"

        if self.ctx[PlatformParameter.name] == _PLATFORM_TOUCH:
            url += "/touch"

        return url

    def _make_test_task(self, urls_resource_id, serps_resource_id):
        sub_ctx = {
            test_task.UrlsResourceParameter.name: urls_resource_id,
            test_task.SerpsResourceParameter.name: serps_resource_id
        }
        sub_task = self.create_subtask(
            task_type=test_task.ImagesTestPumpkinSerp.type,
            input_parameters=sub_ctx,
            description=self.descr,
            execution_space=self.execution_space
        )
        return sub_task.id


__Task__ = ImagesBuildPumpkinSerp
