import json

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.images.metasearch import task as metasearch_task
from sandbox.projects.common.search import components as search_components
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.images.basesearch import components as images_basesearch_components
from sandbox.projects.images.resources import ImagesGenerateBasesearchRequests as img_gen_basesearch_requests


BASESEARCH_PARAMS = metasearch_task.create_basesearch_params(use_fusion=True)
SNIPPETIZER_PARAMS = metasearch_task.create_snippetizer_params(use_fusion=True)


class ImagesGenerateRtyserverRequests(img_gen_basesearch_requests.ImagesGenerateBasesearchRequests):
    """
        Shoot rtyserver basesearch via middlesearch and extract basesearch requests from eventlog
    """

    type = 'IMAGES_GENERATE_RTYSERVER_REQUESTS'

    metasearch_task_input_parameters = metasearch_task.MIDDLESEARCH_PARAMS.params + \
        BASESEARCH_PARAMS.params + SNIPPETIZER_PARAMS.params + \
        (metasearch_task.EnableAppHostParameter,)

    static_input_parameters = img_gen_basesearch_requests.ImagesGenerateBasesearchRequests.static_input_parameters
    shoot_input_parameters = img_gen_basesearch_requests.ImagesGenerateBasesearchRequests.shoot_input_parameters

    input_parameters = \
        static_input_parameters + metasearch_task_input_parameters + shoot_input_parameters

    execution_space = 50 * 1024
    required_ram = 50 * 1024

    def on_execute(self):
        self._substitute_metasearch_task_params()
        return img_gen_basesearch_requests.ImagesGenerateBasesearchRequests.on_execute(self)

    # Override to call fusion search creator.
    def get_imgsearch_impl(self, *args, **kwargs):
        return images_basesearch_components.get_quick_imgsearch(*args, **kwargs)

    def _substitute_metasearch_task_params(self):
        metasearch_task.BASESEARCH_PARAMS = BASESEARCH_PARAMS
        metasearch_task.SNIPPETIZER_PARAMS = SNIPPETIZER_PARAMS
        metasearch_task.BASESEARCH_TYPE = 'IMAGESQUICK'

    # Override snippetizer port. FusionSearch uses several ports on starting.
    def _get_snippetizer_port(self):
        return search_components.DEFAULT_BASESEARCH_PORT + 5

    # For shard primary key for rtyserver search database consist of shard_instance, shard_timestamp, service_id.
    def _get_shard_instance_keys(self):
        shard_attribute_name = media_settings.RTINDEX_SHARD_INSTANCE_ATTRIBUTE_NAME
        timestamp_attribute_name = media_settings.RTINDEX_SHARD_TIMESTAMP_ATTRIBUTE_NAME
        service_attribute_name = media_settings.RTINDEX_SHARD_SERVICE_ATTRIBUTE_NAME

        basesearch_shard_instance = json.dumps({
            attr_name: self._get_database_attribute(metasearch_task.BASESEARCH_PARAMS, attr_name)
            for attr_name in (shard_attribute_name, timestamp_attribute_name, service_attribute_name)
        })

        snippetizer_shard_instance = json.dumps({
            attr_name: self._get_database_attribute(metasearch_task.SNIPPETIZER_PARAMS, attr_name)
            for attr_name in (shard_attribute_name, timestamp_attribute_name, service_attribute_name)
        })

        return basesearch_shard_instance, snippetizer_shard_instance

    @staticmethod
    def get_shard_attrs_from_plan(plan_id):
        attr_name = media_settings.SHARD_INSTANCE_ATTRIBUTE_NAME
        plan_resource = channel.sandbox.get_resource(plan_id)

        if attr_name not in plan_resource.attributes:
            raise SandboxTaskFailureError(
                "No '{}' attribute in plan resource {}".format(attr_name, plan_resource.id))
        attrs_encoded = plan_resource.attributes[attr_name]
        return json.loads(attrs_encoded)


__Task__ = ImagesGenerateRtyserverRequests
