import logging
import os
import time

from sandbox import sdk2
from sandbox.common.rest import Client as SandboxClient
from sandbox.projects.common.dolbilka import resources as dolbilka_resources
from sandbox.projects.images import resource_types as images_resource_types
from sandbox.projects.images.basesearch2.partition import BaseSearchPartition, BASESEARCH_DEFAULT_PORT
from sandbox.projects.images.common import SearchTaskCommonParameters
from sandbox.projects.images.embedding import EMBEDDING_STORAGE
from sandbox.projects.images.embedding.ImagesRunStandaloneEDaemon import ImagesRunStandaloneEDaemon
from sandbox.projects.images.embedding.task import EDaemonComponentTask
from sandbox.projects.images.metasearch2.task import MiddleSearchTask, MIDDLESEARCH_PORT
from sandbox.projects.images.intsearch.task import IntSearchTask, INTSEARCH_PORT
from sandbox.projects.images.inverted_index.task import PDaemonComponentTask
from sandbox.projects.images import util
from sandbox.projects import resource_types
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sdk2.helpers import subprocess

DEFAULT_RPS = 300


class ImagesGenerateSearchDaemonsRequests(sdk2.Task, MiddleSearchTask, IntSearchTask, BaseSearchPartition, EDaemonComponentTask, PDaemonComponentTask):
    """
        Use this task for generate:
          * Embedding storage requests
          * Inverted index requests
          * Int L1 search requests
          * Basesearch L2 search requests
          * Basesearch L3 factor requests
          * Basesearch snippets requests
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 64*1024

    class Parameters(sdk2.Parameters):
        # Inverted index executable
        config = SearchTaskCommonParameters.Parameters.config()
        index_state = SearchTaskCommonParameters.Parameters.index_state()
        partition_number = SearchTaskCommonParameters.Parameters.partition_number()
        search_l1_models_archive = SearchTaskCommonParameters.Parameters.search_l1_models_archive()
        middlesearch_parameters = MiddleSearchTask.Parameters()
        intsearch_parameters = IntSearchTask.Parameters()
        basesearch_parameters = BaseSearchPartition.Parameters()
        edaemon_parameters = EDaemonComponentTask.Parameters()
        pdaemon_parameters = PDaemonComponentTask.Parameters()
        dexecutor = sdk2.parameters.Resource('d-executor executable', resource_type=dolbilka_resources.DEXECUTOR_EXECUTABLE)
        dplanner = sdk2.parameters.Resource('d-planner executable', resource_type=dolbilka_resources.DPLANNER_EXECUTABLE)
        evlog_planner = sdk2.parameters.Resource('Event log planner executable', resource_type=images_resource_types.IMAGES_EVENTLOG_PLANNER_EXECUTABLE)
        plan = sdk2.parameters.Resource('Middlesearch requests (d-executor plan)', resource_type=resource_types.IMAGES_MIDDLESEARCH_PLAN, required=True)
        url_suffix = sdk2.parameters.String('Middlesearch queries suffix', default='', required=False)
        attributes = sdk2.parameters.String('Set additional attrs to resources (ex.: attr1=v1, attr2=v2)')

    class Context(sdk2.Task.Context):
        __first_run = True

    def on_enqueue(self):
        # Fill index state parameter if it not set
        # Extract index state from d-executor plan attributes
        # TODO: do it
        self.Requirements.disk_space = 32*1024 + 32*1024*len(self.Parameters.basesearch_parameters.basesearch_shard_numbers)

        PDaemonComponentTask.on_enqueue(self)
        EDaemonComponentTask.on_enqueue(self)
        BaseSearchPartition.on_enqueue(self)
        IntSearchTask.on_enqueue(self)
        MiddleSearchTask.on_enqueue(self)

        if self.Context.__first_run is False:
            return

        if self.Parameters.dexecutor is None:
            self.Parameters.dexecutor = sdk2.Resource.find(type=dolbilka_resources.DEXECUTOR_EXECUTABLE,
                                                           attrs=dict(released='stable'),
                                                           state='READY').first()

        if self.Parameters.dplanner is None:
            self.Parameters.dplanner = sdk2.Resource.find(type=dolbilka_resources.DPLANNER_EXECUTABLE,
                                                           attrs=dict(released='stable'),
                                                           state='READY').first()

        if self.Parameters.evlog_planner is None:
            self.Parameters.evlog_planner = sdk2.Resource.find(type=images_resource_types.IMAGES_EVENTLOG_PLANNER_EXECUTABLE,
                                                           attrs=dict(released='stable'),
                                                           state='READY').first()

        self.Context.__first_run = False

    def init_resources(self):
        if self.Parameters.dexecutor is None:
            raise SandboxTaskFailureError("Released stable \"d-executor executable\" not found")
        if self.Parameters.dplanner is None:
            raise SandboxTaskFailureError("Released stable \"d-planner executable\" not found")
        if self.Parameters.evlog_planner is None:
            raise SandboxTaskFailureError("Released stable \"Event log planner executable\" not found")

    def on_execute(self):
        # Initialized all resources (auto search if need)
        PDaemonComponentTask.init_resources(self)
        EDaemonComponentTask.init_resources(self)
        BaseSearchPartition.init_resources(self)
        IntSearchTask.init_resources(self)
        MiddleSearchTask.init_resources(self)
        ImagesGenerateSearchDaemonsRequests.init_resources(self)

        # Preprocessing attributes parameter
        target_attributes = []
        for attribute in self.Parameters.attributes.replace(' ', '').split(','):
            entry = attribute.split('=')
            if len(entry) != 2:
                raise SandboxTaskFailureError("Invalid 'attributes' parameter format format")
            target_attributes.append((entry[0], entry[1]))

        # Run standalone embedding storage and basesearch daemon
        client = SandboxClient()

        # TODO: don't enumerate edaemon_parameters and pdaemon_parameters manually
        embedding_task = ImagesRunStandaloneEDaemon(self,
                                                    description="Standalone E-Daemon for ImagesGenerateSearchDaemonsRequests",
                                                    index_state=self.Parameters.index_state,
                                                    partition_number=self.Parameters.partition_number,
                                                    embedding_storage_executable=self.Parameters.edaemon_parameters.embedding_storage_executable,
                                                    search_l1_models_archive=self.Parameters.search_l1_models_archive,
                                                    embedding_database=self.Parameters.edaemon_parameters.embedding_database,
                                                    use_standalone_pdaemon=self.Parameters.edaemon_parameters.use_standalone_pdaemon,
                                                    inverted_index_executable=self.Parameters.pdaemon_parameters.inverted_index_executable,
                                                    shards_number=self.Parameters.pdaemon_parameters.shards_number,
                                                    use_entire_partition=self.Parameters.pdaemon_parameters.use_entire_partition,
                                                    inverted_index_shards=self.Parameters.pdaemon_parameters.inverted_index_shards).enqueue()

        # Start standalone basesearch for all required shards
        basesearch_tasks = []
        for shard in self.basesearch_databases:
            task = sdk2.Task["IMAGES_RUN_STANDALONE_BASESEARCH"](self,
                                                                 description="Standalone basesearch for ImagesGenerateSearchDaemonsRequests",
                                                                 basesearch_executable_resource_id=self.Parameters.basesearch_parameters.basesearch_executable,
                                                                 basesearch_config_resource_id=self.Parameters.basesearch_parameters.basesearch_config,
                                                                 basesearch_database_resource_id=shard,
                                                                 basesearch_models_archive_resource_id=self.Parameters.basesearch_parameters.basesearch_models_archive).enqueue()
            basesearch_tasks.append(task)

        # Get host address for each basesearch instance
        basesearch_hosts = []
        embedding_host = util.wait_for_context_value(client, embedding_task.id, 'network_address', 7200)
        for task in basesearch_tasks:
            host = util.wait_for_context_value(client, task.id, 'network_address', 7200)
            basesearch_hosts.append(host)

        # Convert host addresses to urls (with port)
        # Use braces [] for IPv6 addresses
        embedding_host = "[{}]:{}".format(embedding_host, EMBEDDING_STORAGE.DEFAULT_PORT)
        basesearch_hosts = map(lambda x: "[{}]:{}".format(x, BASESEARCH_DEFAULT_PORT), basesearch_hosts)

        logging.info("Standalone e-daemon started at host {}".format(embedding_host))
        logging.info("Standalone basesearch started at hosts {}".format(basesearch_hosts))

        IntSearchTask.on_execute(self,
                                 embedding_storage_host=embedding_host,
                                 basesearch_shards_num=self.Parameters.basesearch_parameters.basesearch_shard_numbers,
                                 basesearch_hosts=basesearch_hosts)
        MiddleSearchTask.on_execute(self,
                                    int_l1_host_name="localhost:{}".format(INTSEARCH_PORT))

        util.wait_for_context_value(client, embedding_task.id, 'edaemon_ready', 7200)
        for task in basesearch_tasks:
            util.wait_for_context_value(client, task.id, 'basesearch_ready', 7200)

        dexecutor_executable = str(sdk2.ResourceData(self.Parameters.dexecutor).path)
        dplanner_executable = str(sdk2.ResourceData(self.Parameters.dplanner).path)
        evlog_planner_executable = str(sdk2.ResourceData(self.Parameters.evlog_planner).path)
        plan = str(sdk2.ResourceData(self.Parameters.plan).path)

        # Run d-executor
        process_log = sdk2.helpers.ProcessLog(self, logger=logging.getLogger("d-executor"))
        dexecutor_process_id = subprocess.Popen([dexecutor_executable,
                                                 "-p", plan,
                                                 "-P", str(MIDDLESEARCH_PORT),
                                                 "-H", "localhost",
                                                 "-m", "plan",
                                                 "-R", str(DEFAULT_RPS),
                                                 "--augmenturl", str(self.Parameters.url_suffix),
                                                 "-o", "/dev/null"], stdout=process_log.stdout, stderr=process_log.stdout)
        dexecutor_process_id.communicate()
        if dexecutor_process_id.returncode != 0:
            raise SandboxTaskFailureError("d-executor exit code is {}".format(dexecutor_process_id.returncode))
        logging.info("d-executor finished")

        MiddleSearchTask.create_eventlog_resource(self)
        IntSearchTask.create_eventlog_resource(self)

        # Extract queries from event logs
        output_directory = str(self.path("output"))
        os.mkdir(output_directory)
        process_log = sdk2.helpers.ProcessLog(self, logger=logging.getLogger("evlog_planner"))
        evlog_planner_process_id = subprocess.Popen([evlog_planner_executable,
                                                     "-o", output_directory,
                                                     "--active-shards", str(self.Parameters.basesearch_parameters.basesearch_shard_numbers)[1:-1],
                                                     self.intsearch_eventlog_path], stdout=process_log.stdout, stderr=process_log.stdout)
        evlog_planner_process_id.communicate()
        if evlog_planner_process_id.returncode != 0:
            raise SandboxTaskFailureError("evlog_planner exit code is {}".format(evlog_planner_process_id.returncode))
        logging.info("evlog_planner finished")

        # Aggregate all produces query files
        all_queries = []
        # NOTE: temporary disable factors and snippets request generation
        # query_types = ["searchl2", "factors", "snippets", "embedding"]
        query_types = ["searchl2", "embedding"]
        for type in query_types:
            if type == "embedding":
                shard_name = EDaemonComponentTask.get_shard_name(self, self.Parameters.index_state, self.Parameters.partition_number)
                all_queries.append((type, self.Parameters.partition_number, shard_name))
            else:
                for shard_number in self.Parameters.basesearch_parameters.basesearch_shard_numbers:
                    shard_name = BaseSearchPartition.get_shard_name(self, self.Parameters.index_state, shard_number)
                    all_queries.append((type, shard_number, shard_name))

        # Run d-planner processes (create binary d-executor plan files)
        dplanner_processes = []
        for (type, shard_number, shard_name) in all_queries:
            name = "request-{}-{:03d}".format(type, shard_number)
            src = os.path.join(output_directory, "{}.txt".format(name))
            dst = os.path.join(output_directory, "{}.plan".format(name))

            process_log = sdk2.helpers.ProcessLog(self, logger=logging.getLogger("d-planner {}".format(name)))
            dplanner_processes.append(subprocess.Popen([dplanner_executable,
                                                        "-l", src,
                                                        "-o", dst,
                                                        "-t", "plain",
                                                        "-h", "localhost"], stdout=process_log.stdout, stderr=process_log.stdout))

        # Wait d-planner processes
        for pid in dplanner_processes:
            pid.communicate()
        for pid in dplanner_processes:
            if pid.returncode != 0:
                raise SandboxTaskFailureError("d-planner exit code is {}".format(pid.returncode))
        logging.info("d-planner instances finished")

        # Create resources with plain text queries and binary d-executor plans
        # TODO: put shard name to resource attributes
        for (type, shard_number, shard_name) in all_queries:
            plainTextQueries = os.path.join(output_directory, "request-{}-{:03d}.txt".format(type, shard_number))
            plan = os.path.join(output_directory, "request-{}-{:03d}.plan".format(type, shard_number))

            # attributes = {k.format(type): v for k, v in target_attributes.iteritems()}
            attributes = dict(map(lambda x: (x[0].format(type), x[1]), target_attributes))

            plainTextResource = sdk2.ResourceData(resource_types.PLAIN_TEXT_QUERIES(self,
                                                                                    "mmeta, main, {} requests".format(type),
                                                                                    plainTextQueries,
                                                                                    shard_instance=shard_name,
                                                                                    **attributes))
            planResource = sdk2.ResourceData(resource_types.BASESEARCH_PLAN(self,
                                                                            "mmeta, main, {} requests".format(type),
                                                                            plan,
                                                                            shard_instance=shard_name,
                                                                            **attributes))

            plainTextResource.ready()
            planResource.ready()
