# -*- coding: utf-8 -*-

import logging
import random

import sandbox.sdk2 as sdk2

import sandbox.projects.images.resource_types as images_resource_type

AMMO_FILE_NAME = 'ammo.txt'


class ImagesGenerateThumbRequests(sdk2.Task):
    """
        Вытаскивает id тумбов из access-логов балансера выгруженных в MR
    """

    PLAN_FILE_NAME = "thumb.requests"

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        plan = sdk2.parameters.Resource('Raw resource', required=True,
                                        resource_type=images_resource_type.IMAGES_RAW_REQUESTS)

        with sdk2.parameters.RadioGroup('Scheme (do not work with empty domains)') as scheme:
            scheme.values['http'] = scheme.Value(value='http', default=True)
            scheme.values['https'] = scheme.Value(value='https')

        domains = sdk2.parameters.List('List of domains for plan. Use random for each request. Empty for \
                                        location only plan (e.g. for Yandex.Tank ammo)')

    def _rand_hostname(self):
        return random.choice(self.Parameters.domains) if len(self.Parameters.domains) > 0 else ''

    def _parse_formatter(self, resource_path):
        scheme = '{}://'.format(self.Parameters.scheme)

        with open(resource_path, 'r') as raw_plan:
            for line in raw_plan:
                if not line or len(line) < 1 or not all(ord(char) < 128 for char in line):
                    logging.debug("Malformed line: {}. Skipped".format(line))
                    continue
                yield "{scheme}{host}{id}".format(
                    scheme=scheme,
                    host=self._rand_hostname(),
                    id=line.rstrip('\n')
                )

    def on_enqueue(self):
        images_resource_type.IMAGES_THUMBS_REQUESTS(self, self.Parameters.description, AMMO_FILE_NAME)

    def on_execute(self):
        raw_plan_resource = sdk2.ResourceData(self.Parameters.plan)

        plan_resource = images_resource_type.IMAGES_THUMBS_REQUESTS.find(task_id=self.id).first()

        if hasattr(raw_plan_resource, 'collection'):
            plan_resource.collection = raw_plan_resource.collection
        plan_resource.path.write_bytes(
            "\n".join(
                url for url in self._parse_formatter(str(raw_plan_resource.path))
            ))
