import logging

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters

from sandbox.projects.common import decorators
from sandbox.projects.common import string
from sandbox.projects.common import utils
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.images.metasearch import resources as images_metasearch_resources
from sandbox.projects.images.resources import task as resources_task
from sandbox.projects.images.resources import eventlog as resources_eventlog


class IndexTypeParameter(parameters.SandboxStringParameter):
    name = 'index_type'
    description = 'Index type'
    choices = [('Main', media_settings.INDEX_MAIN), ('Cbir', media_settings.INDEX_CBIR_MAIN)]
    default_value = media_settings.INDEX_MAIN
    do_not_copy = True


class ImagesLoadIntsearchResources(resources_task.ImagesProductionResourcesTask,
                                   resources_task.LoadMetasearchResourcesTask):
    """
        Loads queries, configuration files and rearrange data from production
    """

    type = 'IMAGES_LOAD_INTSEARCH_RESOURCES'

    input_parameters = (IndexTypeParameter,) + resources_task.LoadMetasearchResourcesTask.input_parameters

    def on_enqueue(self):
        resources_task.LoadMetasearchResourcesTask.on_enqueue(self)

        data_attributes = self.ctx[resources_task.DataAttrsParameter.name]
        self._create_config_resource(string.parse_attrs(data_attributes))

        queries_attributes = string.parse_attrs(self.ctx[resources_task.QueriesAttrsParameter.name])
        queries_file = self.__get_queries_file()
        attributes = {k.format(queries_file.variant): v for k, v in queries_attributes.iteritems()}
        self._register_queries_resource(queries_file, attributes)

    def on_execute(self):
        meta_index_type = media_settings.INDEX_INT
        base_index_type = self.ctx[IndexTypeParameter.name]
        archived_instances = set([x.strip().lower() for x in utils.get_or_default(self.ctx, resources_task.InstancesWithArchivedPlan).split(',') if x])
        logging.info('Archived instances {}'.format(','.join(archived_instances)))

        if base_index_type == media_settings.INDEX_CBIR_MAIN:
            meta_index_type = media_settings.INDEX_CBIR_INT

        queries_attributes = self.__get_shard_attributes(base_index_type)
        queries_file = self.__get_queries_file()

        for middle_instance in self._get_instances(meta_index_type, archived_instances=archived_instances):
            logging.info('Using instance {}'.format(middle_instance))
            try:
                self._load_config(middle_instance)
                archived_plan = ''
                if middle_instance[0] in archived_instances:
                    logging.info('Try to search archived plan for instance {}'.format(middle_instance[0]))
                    archived_plan = utils.get_and_check_last_resource_with_attribute(
                        images_metasearch_resources.IMAGES_MIDDLESEARCH_PLAN_ARCHIVE,
                        attr_name='instance', attr_value=middle_instance[0].lower())
                else:
                    logging.info('No archived plan for instance {}'.format(middle_instance[0]))
                self._load_queries(middle_instance, meta_index_type, [queries_file], archived_plan)
            except Exception as e:
                logging.error("Failed to acquire queries from {}: {}".format(middle_instance, str(e)))
            else:
                break
        else:
            raise errors.SandboxTaskFailureError("Failed to acquire production data. See logs for details")

        self.__set_shard_attributes(queries_file, queries_attributes)

    def __get_shard_attributes(self, index_type):
        attributes = self._get_intsearch_shard_attributes(index_type)

        # ensure that database available
        snippetizer_index_type = self._get_basesearch_snippetizer_index_type(index_type)
        self._get_basesearch_database(index_type, attributes[media_settings.SHARD_INSTANCE_ATTRIBUTE_NAME])
        self._get_basesearch_database(
            snippetizer_index_type, attributes[media_settings.SNIP_SHARD_INSTANCE_ATTRIBUTE_NAME]
        )
        return attributes

    def __set_shard_attributes(self, queries_file, attributes):
        utils.set_resource_attributes(self.ctx[queries_file.queries_key], attributes)
        utils.set_resource_attributes(self.ctx[queries_file.plan_key], attributes)

    @decorators.memoize
    def __get_queries_file(self):
        index_type = self.ctx[IndexTypeParameter.name]
        return resources_eventlog.QueriesFile(index_type, media_settings.ImagesSettings.COLLECTION_MAIN)


__Task__ = ImagesLoadIntsearchResources
