import logging
from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common import utils
from sandbox.projects.common.mediasearch import yasm_task
from sandbox.projects.images.metasearch import resources as images_metasearch_resources
from sandbox.projects.images.resources import task as resources_task
from sandbox.projects.images.resources import ImagesLoadMiddlesearchResources as load_middlesearch_task
from sandbox.projects.images.resources import ImagesLoadIntsearchResources as load_intsearch_task


_LOAD_MIDDLESEARCH_TASK_IDS = 'middlesearch_task_ids'
_LOAD_INTSEARCH_TASK_IDS = 'intsearch_task_ids'
_GENERATION_TASK_IDS = "generation_task_ids"


# Fields: middle type, base_type, plan variant, update fuzzing resources
_BASE_INDEXES = (
    # Tier0
    (
        media_settings.INDEX_MIDDLE,
        media_settings.INDEX_MAIN,
        media_settings.ImagesSettings.SERP_TYPE_FILTER_ATTRS,
        False,
    ),
    (
        media_settings.INDEX_MIDDLE,
        media_settings.INDEX_CBIR_MAIN,
        media_settings.ImagesSettings.SERP_TYPE_FILTER_DUPS,
        False,
    ),
    (
        media_settings.INDEX_MIDDLE,
        media_settings.INDEX_CBIR_MAIN,
        media_settings.ImagesSettings.SERP_TYPE_FILTER_SIMILAR,
        False,
    ),
    (
        media_settings.INDEX_MIDDLE,
        media_settings.INDEX_MAIN,
        None,
        True,
    ),
    (
        media_settings.INDEX_MIDDLE,
        media_settings.INDEX_QUICK,
        None,
        True,
    ),
    (
        media_settings.INDEX_MIDDLE,
        media_settings.INDEX_CBIR_MAIN,
        None,
        True,
    ),
    (
        media_settings.INDEX_MIDDLE_RQ,
        media_settings.INDEX_RQ,
        None,
        True,
    ),

    # Garbage index
    (
        media_settings.INDEX_MIDDLE,
        media_settings.INDEX_GARBAGE,
        media_settings.ImagesSettings.SERP_TYPE_FILTER_ATTRS,
        False,
    ),
    (
        media_settings.INDEX_MIDDLE,
        media_settings.INDEX_CBIR_GARBAGE,
        media_settings.ImagesSettings.SERP_TYPE_FILTER_DUPS,
        False,
    ),
    (
        media_settings.INDEX_MIDDLE,
        media_settings.INDEX_CBIR_GARBAGE,
        media_settings.ImagesSettings.SERP_TYPE_FILTER_SIMILAR,
        False,
    ),
    (
        media_settings.INDEX_MIDDLE,
        media_settings.INDEX_GARBAGE,
        None,
        True,
    ),
    (
        media_settings.INDEX_MIDDLE,
        media_settings.INDEX_CBIR_GARBAGE,
        None,
        True,
    )
)

RTINDEX_QUICK = (
    media_settings.INDEX_MIDDLE,
    media_settings.INDEX_QUICK,
    None,
    True,
)


class ImagesLoadTestenvData(yasm_task.YasmTask,
                            resources_task.ImagesProductionResourcesTask,
                            resources_task.GenerateResourcesTask):
    """
        Load requests from production for tests and acceptance
    """

    type = "IMAGES_LOAD_TESTENV_DATA"
    push_signal_name = "load_testenv_data"
    execution_space = 180 * 1024

    input_parameters = \
        (resources_task.LoadRequestCountParameter, resources_task.InstancesWithArchivedPlan,) + \
        resources_task.GenerateResourcesTask.input_parameters

    def on_execute(self):
        if _LOAD_MIDDLESEARCH_TASK_IDS not in self.ctx:
            self.ctx[_LOAD_MIDDLESEARCH_TASK_IDS] = {
                meta_index_type: self._load_middlesearch_resources(meta_index_type)
                for meta_index_type in (media_settings.INDEX_MIDDLE, media_settings.INDEX_MIDDLE_RQ)
            }

        if _LOAD_INTSEARCH_TASK_IDS not in self.ctx:
            self.ctx[_LOAD_INTSEARCH_TASK_IDS] = [
                self._load_intsearch_resources(base_index_type)
                for base_index_type in (media_settings.INDEX_MAIN, media_settings.INDEX_CBIR_MAIN)
            ]

        if _GENERATION_TASK_IDS not in self.ctx:
            self.ctx[_GENERATION_TASK_IDS] = {
                _make_ctx_key(base_index_type, serp_type): self._generate_resources(
                    meta_index_type,
                    base_index_type,
                    serp_type,
                    update_fuzzing_resources
                )
                for meta_index_type, base_index_type, serp_type, update_fuzzing_resources in _BASE_INDEXES
            }

            media_settings.ImagesSettings.USE_SAAS_QUICK = True
            meta_index_type, base_index_type, serp_type, update_fuzzing_resources = RTINDEX_QUICK
            self.ctx[_GENERATION_TASK_IDS].update({
                _make_ctx_key(base_index_type, serp_type, 'rtindex'): self._generate_resources(
                    meta_index_type,
                    base_index_type,
                    serp_type,
                    update_fuzzing_resources
                )
            })
            media_settings.ImagesSettings.USE_SAAS_QUICK = False

        utils.check_subtasks_fails(fail_on_first_failure=True)
        self._yasm_notify()

    def _load_intsearch_resources(self, base_index_type):
        meta_index_type = media_settings.INDEX_INT

        sub_ctx = {
            "kill_timeout": self.ctx["kill_timeout"],
            resources_task.LoadRequestCountParameter.name: self.ctx[resources_task.LoadRequestCountParameter.name],
            resources_task.InstancesWithArchivedPlan.name: utils.get_or_default(self.ctx, resources_task.InstancesWithArchivedPlan),
            resources_task.MinRequestCountParameter.name: 'yandsearch=1000',
            resources_task.DataAttrsParameter.name: _make_resource_attributes(meta_index_type, base_index_type),
            resources_task.QueriesAttrsParameter.name: _make_middlesearch_query_attributes(meta_index_type),

            load_intsearch_task.IndexTypeParameter.name: base_index_type,
        }
        sub_task = self.create_subtask(
            task_type=load_intsearch_task.ImagesLoadIntsearchResources.type,
            input_parameters=sub_ctx,
            description="{}, {}".format(self.descr, base_index_type),
            execution_space=self.execution_space,
            inherit_notifications=True
        )
        return sub_task.id

    def _load_middlesearch_resources(self, meta_index_type):
        if meta_index_type == media_settings.INDEX_MIDDLE:
            min_requests = 'yandsearch=1000,quick=1000,imgscbir=10'
        else:
            min_requests = 'yandsearch=1000'

        sub_ctx = {
            "kill_timeout": self.ctx["kill_timeout"],
            resources_task.LoadRequestCountParameter.name: self.ctx[resources_task.LoadRequestCountParameter.name],
            resources_task.InstancesWithArchivedPlan.name: utils.get_or_default(self.ctx, resources_task.InstancesWithArchivedPlan),
            resources_task.MinRequestCountParameter.name: min_requests,
            resources_task.DataAttrsParameter.name: _make_resource_attributes(meta_index_type),
            resources_task.QueriesAttrsParameter.name: _make_middlesearch_query_attributes(meta_index_type),
            load_middlesearch_task.IndexTypeParameter.name: meta_index_type,
        }
        sub_task = self.create_subtask(
            task_type=load_middlesearch_task.ImagesLoadMiddlesearchResources.type,
            input_parameters=sub_ctx,
            description="{}, {}".format(self.descr, meta_index_type),
            execution_space=self.execution_space,
            inherit_notifications=True
        )
        return sub_task.id

    def _get_basesearch_attributes(self, index_type, serp_type):
        return media_settings.ImagesSettings.testenv_basesearch_queries_attributes(index_type, serp_type=serp_type)

    def _get_middlesearch_config(self, index_type=media_settings.INDEX_MIDDLE):
        attribute_name, attribute_value = media_settings.ImagesSettings.testenv_resource_attributes(
            index_type
        )
        return apihelpers.get_task_resource_id(
            task_id=self.ctx[_LOAD_MIDDLESEARCH_TASK_IDS][index_type],
            resource_type=images_metasearch_resources.IMAGES_MIDDLESEARCH_CONFIG,
            attribute_name=attribute_name,
            attribute_value=attribute_value,
        )

    def _get_middlesearch_data(self, index_type=media_settings.INDEX_MIDDLE):
        attribute_name, attribute_value = media_settings.ImagesSettings.testenv_resource_attributes(
            index_type
        )
        return apihelpers.get_task_resource_id(
            task_id=self.ctx[_LOAD_MIDDLESEARCH_TASK_IDS][index_type],
            resource_type=images_metasearch_resources.IMAGES_MIDDLESEARCH_DATA,
            attribute_name=attribute_name,
            attribute_value=attribute_value,
        )

    def _get_middlesearch_plan(self, meta_index_type, base_index_type, serp_type):
        attribute_name, attribute_value = media_settings.ImagesSettings.testenv_middlesearch_queries_attributes(
            meta_index_type,
            base_index_type,
            serp_type
        )
        return apihelpers.get_task_resource_id(
            task_id=self.ctx[_LOAD_MIDDLESEARCH_TASK_IDS][meta_index_type],
            resource_type=resource_types.IMAGES_MIDDLESEARCH_PLAN,
            attribute_name=attribute_name,
            attribute_value=attribute_value,
        )


def _make_middlesearch_query_attributes(middle_type, index_type=None):
    """
        Returns query attributes as a string
    """

    attr_key, attr_value = media_settings.ImagesSettings.testenv_middlesearch_queries_attributes(middle_type, index_type)
    return "{}={}".format(attr_key, attr_value)


def _make_resource_attributes(middle_type, index_type=media_settings.INDEX_MAIN):
    """
        Returns resource attributes as a string
    """

    attr_key, attr_value = media_settings.ImagesSettings.testenv_resource_attributes(middle_type, index_type)
    return "{}={}".format(attr_key, attr_value)


def _make_ctx_key(*args):
    return ",".join(str(a) for a in args)


__Task__ = ImagesLoadTestenvData
