import hashlib
import logging
import os.path
import re
import tarfile

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk import svn

from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common import fuzzing as common_fuzzing


_SVN_USER = "robot-images-sandbox"
_SVN_VAULT_KEY = "ssh_key_for_svn"

#  Section was renamed in DEVTOOLS-3240
_TEST_DATA_SECTION_RE = re.compile(r"DATA\((.*?)\)", flags=re.DOTALL)
_SANDBOX_RESOURCE_RE = re.compile(r"sbr://(\d+)")

_CORPUS_DIR = "corpus"
_CORPUS_ARCHIVE = "generated_corpus.tar"
_CORPUS_RESOURCE_ID = "corpus_resource_id"
_CORPUS_SIZE = 500  # We don't need to generate too large corpus, otherwise, we have no time for fuzzing

_FUZZER_DIR = "fuzzer"


class UpdateFuzzingResourcesParameter(parameters.SandboxBoolParameter):
    name = 'update_fuzzing_resources'
    description = 'Update fuzzing resources'
    default_value = False


class GenerateFuzzingTask:
    """Mixin to generate fuzzing corpus"""

    input_parameters = (UpdateFuzzingResourcesParameter,)

    def on_enqueue(self):
        self.ctx[_CORPUS_RESOURCE_ID] = self.create_resource(
            self.descr,
            _CORPUS_ARCHIVE,
            resource_types.FUZZ_CORPUS_DATA,
            arch=sandboxapi.ARCH_ANY,
        ).id

    def _fuzzing_dir(self):
        raise NotImplementedError()

    def _fuzzing_corpus_init(self):
        paths.make_folder(_CORPUS_DIR, delete_content=True)
        self.__corpus_size = 0

    def _fuzzing_corpus_add(self, query, max_corpus_size=_CORPUS_SIZE):
        if self.__corpus_size >= max_corpus_size:
            return

        hasher_md5 = hashlib.md5()
        hasher_md5.update(query)
        corpus_item = hasher_md5.hexdigest()
        with open(os.path.join(_CORPUS_DIR, corpus_item), "w") as corpus_file:
            corpus_file.write(query)

        self.__corpus_size += 1

    def _fuzzing_corpus_finish(self, *sandbox_resource_ids):
        """
            Creates archive and register it in fuzzing infrastructure

            Also update sandbox resources in test dir if needed
        """
        with tarfile.open(_CORPUS_ARCHIVE, "w") as tar:
            tar.add(_CORPUS_DIR, ".")

        if not utils.get_or_default(self.ctx, UpdateFuzzingResourcesParameter):
            return

        corpus_resource_id = self.ctx[_CORPUS_RESOURCE_ID]
        self.mark_resource_ready(corpus_resource_id)

        target_dir = svn.Arcadia.checkout(svn.Arcadia.trunk_url(self._fuzzing_dir()), _FUZZER_DIR)
        target_file = os.path.join(target_dir, 'ya.make')

        with open(target_file) as f:
            logging.info("Target file before: '{}'".format(f.read()))

        if sandbox_resource_ids:
            self.__update_yamake_resources(target_file, *sandbox_resource_ids)

        with open(target_file) as f:
            logging.info("Target file after: '{}'".format(f.read()))

        with ssh.Key(self, key_owner=_SVN_USER, key_name=_SVN_VAULT_KEY):
            common_fuzzing.commit_data_with_corpus(target_file, corpus_resource_id, self._fuzzing_dir(), _SVN_USER, description="SKIP_CHECK")

    def __update_yamake_resources(self, yamake_path, first_resource_id, *rest_resource_ids):
        """
            Updates all sandbox resources inside TEST_DATA section of ya.make
            All sandbox resources inside TEST_DATA section will be replaced with specified identifiers
        """

        new_resources = (first_resource_id,) + rest_resource_ids

        with open(yamake_path) as yamake_file:
            yamake_data = yamake_file.read()

        test_data_match = _TEST_DATA_SECTION_RE.search(yamake_data)
        if not test_data_match:
            raise errors.SandboxTaskFailureError("Failed to find TEST_DATA section inside specified resource")

        old_test_data = test_data_match.group(1)
        old_resources = list(_SANDBOX_RESOURCE_RE.finditer(old_test_data))
        if len(old_resources) != len(new_resources):
            raise errors.SandboxTaskFailureError("Different number of old and new sandbox resources")

        pos = 0
        new_test_data = ""
        for old_match, new_resource_id in zip(old_resources, new_resources):
            new_test_data += old_test_data[pos:old_match.start(1)] + str(new_resource_id)
            pos = old_match.end(1)
        new_test_data += old_test_data[pos:]

        yamake_data = yamake_data[:test_data_match.start(1)] + new_test_data + yamake_data[test_data_match.end(1):]

        with open(yamake_path, "w") as yamake_file:
            yamake_file.write(yamake_data)
