import logging
import random
import os.path
import urllib
import re

import sandbox.common.types.client as ctc

from sandbox.common import fs, share, errors

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types

from sandbox.projects.common import apihelpers
from sandbox.projects.common import dolbilka
from sandbox.projects.common import decorators
from sandbox.projects.common import utils
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.search import database as search_database
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.images import ImagesLoadBasesearchDatabase as load_task
from sandbox.projects.images.idx_ops import resources as images_idx_ops_resources
from sandbox.projects.images.metasearch import resources as images_metasearch_resources
from sandbox.projects.images.metasearch import task as metasearch_task
from sandbox.projects.images.models import resources as images_models_resources
from sandbox.projects.images.resources import ImagesGenerateBasesearchRequests as generate_task
from sandbox.projects.images.resources import ImagesGenerateRtyserverRequests as generate_quick_task
from sandbox.projects.images.resources import eventlog as resources_eventlog
from sandbox.projects.images.resources import fuzzing as resources_fuzzing
from sandbox.projects.images.daemons import resources as daemons_resources

from sandbox.sandboxsdk.copy import RemoteCopy

_EVENTLOG_CURR_PATH = 'current-eventlog-{meta_type}-{port}'
_EVENTLOG_PREV_PATH = 'eventlog-{meta_type}-{port}.PREV'
_EVENTLOG_PREV_PATH_NEW = 'current-eventlog-{meta_type}-{port}.1'
_EVENTLOG_URL_PREFIX = "rsync://{host}/logs"
_EVENTLOG_URL_PREFIX_SKYNET = "{host}:/usr/local/www/logs"

_CONFIG_RESOURCE_ID = 'out_config_resource_id'
_QUERIES_RESOURCE_ID = 'out_{}_queries_resource_id'
_PLAN_RESOURCE_ID = 'out_{}_plan_resource_id'
_APPHOST_QUERIES_RESOURCE_ID = 'apphost_{}_queries_resource_id'
_APPHOST_PLAN_RESOURCE_ID = 'apphost_{}_plan_resource_id'

_MAX_TRIES = 6


class GenerationRequestCountParameter(parameters.SandboxIntegerParameter):
    name = "generation_request_count"
    description = "Requests for generation"
    default_value = 400000


class LoadRequestCountParameter(parameters.SandboxIntegerParameter):
    """
        Maximum number of queries to read
    """

    name = 'request_count'
    description = 'Maximum number of requests'
    default_value = 10000


class MinRequestCountParameter(parameters.SandboxStringParameter):
    """
        Minumum number of queries to use
    """

    name = 'min_request_count'
    description = 'Minumum number of requests'
    default_value = "yandsearch=1000,quick=1000,imagesultra=1000,imgscbir=10"


class LogNameParameter(parameters.SandboxStringParameter):
    """
        Pattern of eventlog name
    """

    name = 'log_name'
    description = 'Mmeta/Int log name (blank for default)'


class InstancesWithArchivedPlan(parameters.SandboxStringParameter):
    """
        Use prepared sbr instead of sky-sharing eventlog from instances
    """

    name = 'instances_with_archived_plan'
    description = 'Instances(hosts) with archived plan  sas1-4369.search.yandex.net,sas2-7150.search.yandex.net'
    default_value = ''


class GenerateResourcesTask(task.SandboxTask):
    """Base class with methods to generate basesearch plans"""

    input_parameters = (
        GenerationRequestCountParameter,
    )

    # basesearch
    def _get_basesearch_executable(self, index_type):
        raise NotImplementedError()

    def _get_basesearch_config(self, index_type):
        raise NotImplementedError()

    def _get_basesearch_models(self, index_type):
        raise NotImplementedError()

    def _get_basesearch_database(self, index_type):
        raise NotImplementedError()

    def _get_basesearch_snippetizer_index_type(self, index_type):
        raise NotImplementedError()

    def _get_basesearch_attributes(self, index_type, serp_type):
        raise NotImplementedError()

    # middlesearch
    def _get_middlesearch_executable(self, index_type):
        raise NotImplementedError()

    def _get_middlesearch_config(self, index_type):
        raise NotImplementedError()

    def _get_middlesearch_data(self, index_type):
        raise NotImplementedError()

    def _get_middlesearch_index(self):
        raise NotImplementedError()

    def _get_middlesearch_models(self):
        raise NotImplementedError()

    def _get_middlesearch_plan(self, meta_index_type, base_index_type, serp_type):
        raise NotImplementedError()

    def _get_basesearch_ctx_params(self, params_obj, index_type):
        if index_type == media_settings.INDEX_QUICK and media_settings.ImagesSettings.USE_SAAS_QUICK:
            logging.info('Building basesearch ctx params for rt-index quick.')
            return {
                params_obj.Binary.name: self._get_basesearch_executable(index_type),
                params_obj.ConfigsFolder.name: self._get_basesearch_config(index_type),
                params_obj.Database.name: self._get_basesearch_database(index_type),
                params_obj.StaticData.name: media_settings.ImagesSettings.get_rtyserver_static_data(),
                params_obj.ShardWriterConfig.name: media_settings.ImagesSettings.get_rtyserver_sw_config(),
            }
        else:
            return {
                params_obj.Binary.name: self._get_basesearch_executable(index_type),
                params_obj.Config.name: self._get_basesearch_config(index_type),
                params_obj.Database.name: self._get_basesearch_database(index_type),
            }

    def _generate_resources(self, meta_index_type, base_index_type,
                            serp_type=None, update_fuzzing_resources=False, enable_apphost=False):
        """
            Generate plans for basesearch
        """

        snippetizer_index_type = self._get_basesearch_snippetizer_index_type(base_index_type)
        if base_index_type == media_settings.INDEX_QUICK and media_settings.ImagesSettings.USE_SAAS_QUICK:
            generate_task_impl = generate_quick_task
            generate_task_type_impl = generate_quick_task.ImagesGenerateRtyserverRequests.type
        else:
            generate_task_impl = generate_task
            generate_task_type_impl = generate_task.ImagesGenerateBasesearchRequests.type

        sub_ctx = media_settings.ImagesSettings.basesearch_shooting_parameters(meta_index_type)
        sub_ctx.update({
            metasearch_task.MIDDLESEARCH_PARAMS.Binary.name: self._get_middlesearch_executable(meta_index_type),
            metasearch_task.MIDDLESEARCH_PARAMS.Config.name: self._get_middlesearch_config(meta_index_type),
            metasearch_task.MIDDLESEARCH_PARAMS.Data.name: self._get_middlesearch_data(meta_index_type),
            metasearch_task.EnableAppHostParameter.name: enable_apphost,

            generate_task.PlanParameter.name: self._get_middlesearch_plan(meta_index_type, base_index_type, serp_type),
            generate_task.AttributesParameter.name: "{}={}".format(*self._get_basesearch_attributes(base_index_type, serp_type)),
            resources_fuzzing.UpdateFuzzingResourcesParameter.name: update_fuzzing_resources,
        })

        sub_ctx.update(
            self._get_basesearch_ctx_params(generate_task_impl.BASESEARCH_PARAMS, base_index_type)
        )
        sub_ctx.update(
            self._get_basesearch_ctx_params(generate_task_impl.SNIPPETIZER_PARAMS, snippetizer_index_type)
        )

        if meta_index_type == media_settings.INDEX_MIDDLE:
            sub_ctx.update({
                metasearch_task.MIDDLESEARCH_PARAMS.ArchiveModel.name: self._get_middlesearch_models(),
                metasearch_task.MIDDLESEARCH_PARAMS.Index.name: self._get_middlesearch_index(),
            })

        basesearch_models = self._get_basesearch_models(base_index_type)
        if basesearch_models:
            sub_ctx[generate_task_impl.BASESEARCH_PARAMS.ArchiveModel.name] = basesearch_models

        snippetizer_models = self._get_basesearch_models(snippetizer_index_type)
        if snippetizer_models:
            sub_ctx[generate_task_impl.SNIPPETIZER_PARAMS.ArchiveModel.name] = snippetizer_models

        description = "{}, {}, {}".format(self.descr, meta_index_type, base_index_type)
        if serp_type is not None:
            description = "{}, {}".format(description, serp_type)

        sub_task = self.create_subtask(
            task_type=generate_task_type_impl,
            description=description,
            input_parameters=sub_ctx,
            arch=sandboxapi.ARCH_LINUX,
            inherit_notifications=True
        )
        return sub_task.id

    def _generated_plan_id(self, generation_task_id, query_type):
        generation_task = channel.sandbox.get_task(generation_task_id)
        return generation_task.ctx[generate_task.PLAN_RESOURCE_KEY.format(query_type)]

    def _basesearch_database_id(self, generation_task_id, index_type):
        generation_task = channel.sandbox.get_task(generation_task_id)
        if index_type == media_settings.INDEX_QUICK and media_settings.ImagesSettings.USE_SAAS_QUICK:
            return generation_task.ctx[generate_quick_task.BASESEARCH_PARAMS.Database.name]
        else:
            return generation_task.ctx[generate_task.BASESEARCH_PARAMS.Database.name]


class QueriesAttrsParameter(parameters.SandboxStringParameter):
    name = 'queries_attributes'
    description = 'Create and set attrs to queries (e.g. attr1=v1, attr2=v2)'
    default_value = "debug_{}=yes"
    do_not_copy = True


class DataAttrsParameter(parameters.SandboxStringParameter):
    name = 'data_attributes'
    description = 'Create and set attrs to data (e.g. attr1=v1, attr2=v2)'
    default_value = "debug=yes"
    do_not_copy = True


class RemoteCopyHeadSkynet(RemoteCopy):

    @classmethod
    def compatible(cls, src, parsed_url):
        return True

    def __call__(self, files=None, head=500000, user=None, **kws):
        host, _, srcdir = self._src.partition(":")
        share.skynet_run_and_copy(host, srcdir, self._dst, files=files, method=share.ShareAndCopyHeadFiles(user, files, srcdir, head), user=user)


class LoadMetasearchResourcesTask(task.SandboxTask):

    input_parameters = (
        QueriesAttrsParameter,
        DataAttrsParameter,
        LoadRequestCountParameter,
        MinRequestCountParameter,
        LogNameParameter,
        InstancesWithArchivedPlan,
    )

    client_tags = ctc.Tag.Group.LINUX

    def _get_middlesearch_instances(self, index_type, is_hamster=False):
        raise NotImplementedError()

    def _create_config_resource(self, attributes, is_hamster=False):
        config_resource = self.create_resource(
            self.descr,
            self._get_config_path(is_hamster),
            images_metasearch_resources.IMAGES_MIDDLESEARCH_CONFIG,
            arch=sandboxapi.ARCH_ANY,
            attributes=attributes
        )
        self.ctx[_CONFIG_RESOURCE_ID] = config_resource.id

    def _create_queries_resources(self, collection_name, attributes):
        """
            Creates dexecutor plan and plain text queries for specified collection
        """

        description = '{}, queries to collection \'{}\''.format(self.descr, collection_name)

        self.ctx[self._get_queries_key(collection_name)] = self.create_resource(
            description,
            self._get_queries_path(collection_name),
            resource_types.IMAGES_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
            arch=sandboxapi.ARCH_ANY,
            attributes=attributes
        ).id
        self.ctx[self._get_plan_key(collection_name)] = self.create_resource(
            description,
            self._get_plan_path(collection_name),
            resource_types.IMAGES_MIDDLESEARCH_PLAN,
            arch=sandboxapi.ARCH_ANY,
            attributes=attributes
        ).id

    def _register_queries_resource(self, queries_file, attributes):
        self.ctx[queries_file.queries_key] = self.create_resource(
            "{}, {}".format(self.descr, queries_file.description),
            self.abs_path(queries_file.queries_path),
            queries_file.queries_resource,
            arch=sandboxapi.ARCH_ANY,
            attributes=attributes
        ).id
        self.ctx[queries_file.plan_key] = self.create_resource(
            "{}, {}".format(self.descr, queries_file.description),
            self.abs_path(queries_file.plan_path),
            queries_file.plan_resource,
            arch=sandboxapi.ARCH_ANY,
            attributes=attributes
        ).id

    def _get_queries_path(self, collection_name):
        return self.abs_path('metasearch-{}.queries.txt'.format(collection_name))

    def _get_apphost_queries_path(self, collection_name):
        return self.abs_path('metasearch-apphost-{}.queries.txt'.format(collection_name))

    def _get_plan_path(self, collection_name):
        return self.abs_path('metasearch-{}.plan.bin'.format(collection_name))

    def _get_apphost_plan_path(self, collection_name):
        return self.abs_path('metasearch-apphost-{}.plan.bin'.format(collection_name))

    def _get_queries_key(self, collection_name):
        return _QUERIES_RESOURCE_ID.format(collection_name)

    def _get_apphost_queries_key(self, collection_name):
        return _APPHOST_QUERIES_RESOURCE_ID.format(collection_name)

    def _get_plan_key(self, collection_name):
        return _PLAN_RESOURCE_ID.format(collection_name)

    def _get_apphost_plan_key(self, collection_name):
        return _APPHOST_PLAN_RESOURCE_ID.format(collection_name)

    def _get_config_path(self, is_hamster=False):
        return "metasearch_hamster.cfg" if is_hamster else "metasearch.cfg"

    def _get_instances(self, index_type, is_hamster=False, archived_instances={}):
        """
            Returns random metasearch instances
        """

        def ping_instance(host_and_port):
            try:
                logging.info('Pinging instance {}'.format(host_and_port))
                urllib.urlopen('http://{}:{}/yandsearch'.format(host_and_port[0], host_and_port[1]))
                return True
            except IOError:
                logging.info('Pinging instance {} failed'.format(host_and_port))
                return False

        def ping_instance_by_container_hostname(host_port_container):
            try:
                logging.info('Pinging instance {}'.format(host_port_container))
                urllib.urlopen('http://{}:{}/yandsearch'.format(host_port_container[2], host_port_container[1]))
                return True
            except IOError:
                logging.info('Pinging instance by container hostname {} failed'.format(host_port_container))
                return False

        hosts = self._get_middlesearch_instances(index_type, is_hamster)
        logging.info("metasearch hosts={}".format(hosts))
        if not hosts:
            raise errors.TaskFailure("No hosts found for index_type={}{}".format(index_type, " from hamster" if is_hamster else ""))

        sample_size = 20

        if archived_instances:
            priority_hosts = [x for x in hosts if x[0].lower() in archived_instances or x[2].lower() in archived_instances]
            secondary_hosts = [x for x in hosts if x[0].lower() not in archived_instances and x[2].lower() not in archived_instances]
            random.shuffle(secondary_hosts)
            hosts = (priority_hosts + secondary_hosts)[:sample_size]
        elif len(hosts) > sample_size:
            hosts = random.sample(hosts, sample_size)

        live_hosts = [(h[0], h[1]) for h in hosts if ping_instance(h)]
        if not live_hosts:
            live_hosts_by_container_hostname = [(h[2], h[1]) for h in hosts if ping_instance_by_container_hostname(h)]
            if not live_hosts_by_container_hostname:
                raise errors.TaskFailure("Could not find any working instance for index_type={}{}".format(index_type, " from hamster" if is_hamster else ""))

        return live_hosts[:_MAX_TRIES] if live_hosts else live_hosts_by_container_hostname[:_MAX_TRIES]

    def _load_config(self, middle_instance, is_hamster=False):
        utils.get_config_by_info_request(middle_instance[0], middle_instance[1], self._get_config_path(is_hamster))

    def _load_queries(self, middle_instance, meta_index_type, queries_files, archived_plan):
        utils.receive_skynet_key(self.owner)
        skynet_user = "robot-images"
        if self.owner.islower():
            skynet_user = self.owner

        host, port = middle_instance

        if self.ctx[LogNameParameter.name]:
            log_names = (self.ctx[LogNameParameter.name],)
        else:
            if meta_index_type == media_settings.INDEX_MIDDLE_RQ:
                log_names = (_EVENTLOG_CURR_PATH, _EVENTLOG_PREV_PATH_NEW)  # IMGDEVOPS-1819
            else:
                log_names = (_EVENTLOG_CURR_PATH, _EVENTLOG_PREV_PATH)

        for queries_file in queries_files:
            queries_file.open(self)

        max_requests_count = self.ctx[LoadRequestCountParameter.name]
        requests_count = 0

        meta_type = "int" if meta_index_type in [media_settings.INDEX_INT, media_settings.INDEX_CBIR_INT] else "mmeta"

        try:
            for log_name in log_names:
                file_name = log_name.format(
                    meta_type=meta_type,
                    port=port
                )
                local_path = self.abs_path(file_name)
                try:
                    if archived_plan:
                        archived_plan_path = self.sync_resource(archived_plan)
                        logging.info('Copy {} to {}'.format(archived_plan_path, local_path))
                        paths.copy_path(archived_plan_path, local_path)
                        logging.info('Using archived plan')
                    else:
                        remote_url = _EVENTLOG_URL_PREFIX_SKYNET.format(
                            host=host
                        )
                        RemoteCopyHeadSkynet(remote_url, os.path.dirname(local_path))(files=[file_name], head=500000, user=skynet_user)

                    with resources_eventlog.Evlogdump(local_path) as evlogdump:
                        for data in resources_eventlog.read_eventlog(evlogdump):
                            for queries_file in queries_files:
                                queries_file.write(data)

                            requests_count += 1
                            if requests_count >= max_requests_count:  # break inner loop
                                break
                finally:
                    fs.remove_path_safely(local_path)

                if requests_count >= max_requests_count:  # break outer loop
                    break
        finally:
            for queries_file in queries_files:
                queries_file.close()

        min_counters = {}
        for item in utils.get_or_default(self.ctx, MinRequestCountParameter).split(","):
            key, value = item.split("=", 1)
            min_counters[key] = int(value)

        for queries_file in queries_files:
            queries_file.validate(min_counters)
            dolbilka.convert_queries_to_plan(
                queries_file.queries_path,
                queries_file.plan_path,
                loader_type=queries_file.plan_type
            )


class ImagesProductionResourcesTask:
    """Mixin class to calculate resources from production"""

    @decorators.memoize
    def _get_idx_ops_executable(self):
        return utils.get_and_check_last_released_resource_id(
            images_idx_ops_resources.IDX_OPS_IMAGES_EXECUTABLE,
            arch=sandboxapi.ARCH_LINUX
        )

    @decorators.memoize
    def _get_archiver_executable(self):
        archiver_attributes = media_settings.ImagesSettings.RESOURCE_PRIEMKA_ATTRIBUTES
        return utils.get_and_check_last_resource_with_attribute(
            resource_types.ARCHIVER_TOOL_EXECUTABLE,
            archiver_attributes[0],
            archiver_attributes[1]
        ).id

    @decorators.memoize
    def _get_mx_ops_executable(self):
        mxops_attributes = media_settings.ImagesSettings.RESOURCE_PRIEMKA_ATTRIBUTES
        return utils.get_and_check_last_resource_with_attribute(
            resource_types.MX_OPS_EXECUTABLE,
            mxops_attributes[0],
            mxops_attributes[1]
        )

    def _get_basesearch_executable(self, index_type=media_settings.INDEX_MAIN):
        service_id = media_settings.ImagesSettings.service_id(index_type)
        logging.info('Load basesearch executable from nanny service: {}'.format(service_id))
        return self.__get_nanny_file(
            service_id,
            media_settings.ImagesSettings.basesearch_executable_resource(index_type)
        )

    def _get_basesearch_executable_last_released(self, index_type=media_settings.INDEX_MAIN):
        return utils.get_and_check_last_released_resource_id(
            media_settings.ImagesSettings.basesearch_executable_resource(index_type),
            arch=sandboxapi.ARCH_LINUX
        )

    def _get_basesearch_task(self, index_type=media_settings.INDEX_MAIN):
        return self.__get_nanny_task(
            media_settings.ImagesSettings.service_id(index_type),
            media_settings.ImagesSettings.basesearch_executable_resource(index_type)
        )

    def _get_basesearch_config(self, index_type=media_settings.INDEX_MAIN):
        config_resource = media_settings.ImagesSettings.basesearch_config_resource(index_type)
        try:
            # use fake config for test if exists
            return self.__get_nanny_file(
                media_settings.ImagesSettings.service_id(index_type),
                config_resource
            )
        except errors.TaskFailure as e:
            logging.info(e)
            return self.__get_sandbox_resource(
                self._get_basesearch_task(index_type),
                config_resource
            ).id

    def _get_basesearch_config_last_released(self, index_type=media_settings.INDEX_MAIN):
        return utils.get_and_check_last_released_resource_id(
            media_settings.ImagesSettings.basesearch_config_resource(index_type),
            arch=sandboxapi.ARCH_LINUX
        )

    def _get_basesearch_settings_last_released(self, index_type=media_settings.INDEX_MAIN):
        return utils.get_and_check_last_released_resource_id(
            media_settings.ImagesSettings.basesearch_settings_resource(index_type),
            arch=sandboxapi.ARCH_LINUX
        )

    def _get_basesearch_shard_name(self, index_type=media_settings.INDEX_MAIN):
        # cbirdaemon has a shard without shardmap
        if index_type == media_settings.INDEX_CBIR_DAEMON:
            task_id = self.__get_nanny_task(media_settings.ImagesSettings.service_id(index_type),
                                             daemons_resources.CBIR_DAEMON2_SETTINGS)
            resource = self.__get_sandbox_resource(task_id, daemons_resources.CBIR_DAEMON2_SETTINGS)
            return resource.attributes[media_settings.SHARD_INSTANCE_ATTRIBUTE_NAME]
        else:
            return self.__get_shard_name(index_type, media_settings.ImagesSettings.basesearch_database_prefix(index_type))

    def _get_basesearch_database(self, index_type=media_settings.INDEX_MAIN, shard_name=None):
        # cbirdaemon has a shard without shardmap
        if index_type == media_settings.INDEX_CBIR_DAEMON:
            return self.__get_nanny_file(media_settings.ImagesSettings.service_id(index_type),
                                         daemons_resources.CBIR_DAEMON2_SETTINGS)

        # Frozen shard for quick/ultra.
        if index_type == media_settings.INDEX_QUICK and media_settings.ImagesSettings.USE_SAAS_QUICK:
            service_id = media_settings.ImagesSettings.service_id(index_type)
            attrs = {'service': service_id}
            if shard_name is not None:
                if isinstance(shard_name, dict):
                    attrs.update(shard_name)
                else:
                    attrs[media_settings.RTINDEX_SHARD_INSTANCE_ATTRIBUTE_NAME] = shard_name

            database_resource = apihelpers.get_last_resource_with_attrs(
                resource_types.RTYSERVER_SEARCH_DATABASE,
                attrs,
                all_attrs=True)

            if not database_resource:
                raise errors.TaskFailure(
                    'Failed to find RTYSERVER_SEARCH_DATABASE with service={}.'.format(service_id))
            return database_resource.id

        resource_type = media_settings.ImagesSettings.basesearch_database_resource(index_type)
        if shard_name is None:
            shard_name = self._get_basesearch_shard_name(index_type)

        # search whether resource already available
        database_resource = apihelpers.get_last_resource_with_attribute(
            resource_type,
            media_settings.SHARD_INSTANCE_ATTRIBUTE_NAME,
            shard_name
        )
        if database_resource:
            return database_resource.id

        # load database from production if it doesn't exists
        sub_ctx = {
            load_task.IndexTypeParameter.name: index_type,
            search_database.ShardNameParameter.name: shard_name,
        }

        if index_type == media_settings.INDEX_RQ:
            sub_ctx[search_database.ShardLinkParameter.name] = self.__get_rq_shard_link()

        sub_task = self.create_subtask(
            task_type=load_task.ImagesLoadBasesearchDatabase.type,
            description="{}, {}".format(self.descr, shard_name),
            input_parameters=sub_ctx,
            inherit_notifications=True
        )
        database_resource = apihelpers.list_task_resources(
            sub_task.id,
            resource_type=resource_type,
        )
        if not database_resource:
            raise errors.TaskFailure('Failed to find database resource in load task')
        return database_resource[0].id

    def _get_basesearch_models(self, index_type=media_settings.INDEX_MAIN):
        if index_type in (media_settings.INDEX_MAIN, media_settings.INDEX_QUICK, media_settings.INDEX_ULTRA):
            return self.__get_nanny_file(
                media_settings.ImagesSettings.service_id(index_type),
                images_models_resources.IMAGES_DYNAMIC_MODELS_ARCHIVE
            )
        else:
            return None

    def _get_basesearch_models_task(self, index_type=media_settings.INDEX_MAIN):
        if index_type in (media_settings.INDEX_MAIN, media_settings.INDEX_QUICK, media_settings.INDEX_ULTRA):
            return self.__get_nanny_task(
                media_settings.ImagesSettings.service_id(index_type),
                images_models_resources.IMAGES_DYNAMIC_MODELS_ARCHIVE
            )
        else:
            return None

    def _get_basesearch_snippetizer_index_type(self, index_type):
        return media_settings.ImagesSettings.snippetizer_index_type(index_type)

    @decorators.memoize
    def _get_basesearch_plan(self, index_type, query_type, serp_type=None):
        attribute_name, attribute_value = media_settings.ImagesSettings.testenv_basesearch_queries_attributes(
            index_type,
            query_type,
            serp_type=serp_type
        )
        return utils.get_and_check_last_resource_with_attribute(
            resource_types.BASESEARCH_PLAN,
            attribute_name,
            attribute_value
        ).id

    @decorators.memoize
    def _get_middlesearch_executable(self, index_type=media_settings.INDEX_MIDDLE):
        return self.__get_nanny_file(
            media_settings.ImagesSettings.service_id(index_type),
            resource_types.IMAGES_MIDDLESEARCH_EXECUTABLE
        )

    @decorators.memoize
    def _get_middlesearch_config(self, index_type=media_settings.INDEX_MIDDLE):
        attribute_name, attribute_value = media_settings.ImagesSettings.testenv_resource_attributes(index_type)
        return utils.get_and_check_last_resource_with_attribute(
            images_metasearch_resources.IMAGES_MIDDLESEARCH_CONFIG,
            attribute_name,
            attribute_value
        ).id

    @decorators.memoize
    def _get_middlesearch_data(self, index_type=media_settings.INDEX_MIDDLE):
        attribute_name, attribute_value = media_settings.ImagesSettings.testenv_resource_attributes(
            index_type
        )
        return utils.get_and_check_last_resource_with_attribute(
            images_metasearch_resources.IMAGES_MIDDLESEARCH_DATA,
            attribute_name,
            attribute_value
        ).id

    @decorators.memoize
    def _get_middlesearch_models(self):
        return self.__get_nanny_file(
            media_settings.ImagesSettings.service_id(media_settings.INDEX_MIDDLE),
            images_models_resources.IMAGES_MIDDLESEARCH_DYNAMIC_MODELS_ARCHIVE
        )

    @decorators.memoize
    def _get_middlesearch_shard_name(self):
        index_type = media_settings.INDEX_MIDDLE
        return self.__get_shard_name(index_type, media_settings.ImagesSettings.basesearch_database_prefix(index_type))

    @decorators.memoize
    def _get_intsearch_shard_attributes(self, index_type):
        base_shard_name = self._get_basesearch_shard_name(index_type)
        snip_shard_name = self._get_basesearch_shard_name(self._get_basesearch_snippetizer_index_type(index_type))
        return {
            media_settings.SHARD_INSTANCE_ATTRIBUTE_NAME: base_shard_name,
            media_settings.SNIP_SHARD_INSTANCE_ATTRIBUTE_NAME: snip_shard_name,
        }

    @decorators.memoize
    def _get_middlesearch_shard_attributes(self, index_type):
        attributes = self._get_intsearch_shard_attributes(index_type)
        meta_shard_name = self._get_middlesearch_shard_name()
        attributes.update({
            media_settings.META_SHARD_INSTANCE_ATTRIBUTE_NAME: meta_shard_name,
        })
        return attributes

    @decorators.memoize
    def _get_middlesearch_index(self):
        shard_name = self._get_middlesearch_shard_name()
        return utils.get_and_check_last_resource_with_attribute(
            media_settings.ImagesSettings.basesearch_database_resource(media_settings.INDEX_MIDDLE),
            media_settings.SHARD_INSTANCE_ATTRIBUTE_NAME,
            shard_name
        ).id

    @decorators.memoize
    def _get_middlesearch_plan(self, meta_index_type, base_index_type, serp_type=None):
        attribute_name, attribute_value = media_settings.ImagesSettings.testenv_middlesearch_queries_attributes(
            meta_index_type,
            base_index_type,
            serp_type=serp_type
        )
        return utils.get_and_check_last_resource_with_attribute(
            resource_types.IMAGES_MIDDLESEARCH_PLAN,
            attribute_name,
            attribute_value
        ).id

    @decorators.memoize
    def _get_middlesearch_queries(self, meta_index_type, base_index_type, serp_type=None):
        attribute_name, attribute_value = media_settings.ImagesSettings.testenv_middlesearch_queries_attributes(
            meta_index_type,
            base_index_type,
            serp_type=serp_type
        )
        return utils.get_and_check_last_resource_with_attribute(
            resource_types.IMAGES_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
            attribute_name,
            attribute_value
        ).id

    @decorators.memoize
    def _get_middlesearch_instances(self, index_type, is_hamster=False):
        return self.__get_nanny_instances(media_settings.ImagesSettings.service_id(index_type, is_hamster=is_hamster))

    @decorators.memoize
    def _get_nanny_shard_map_data(self, service_id):
        service_resources = self.__get_nanny_resource(service_id)
        if 'sandbox_bsc_shard' not in service_resources:
            return None
        return service_resources['sandbox_bsc_shard']['sandbox_shardmap']

    @decorators.memoize
    def _get_nanny_cajuper_index_state(self, service_id):
        service_resources = self.__get_nanny_resource(service_id)
        for sandbox_file in service_resources['sandbox_files']:
            if sandbox_file['task_type'] == 'SWITCH_IMAGES_DB_INDEX' and sandbox_file['resource_type'] == 'IMAGES_PRODUCTION_INDEX_STATE_RESOURCE':
                return self.sync_resource(sandbox_file['resource_id'])
        return None

    def __get_rq_shard_link(self):
        shardmap_path = self.__get_nanny_shard_map(media_settings.ImagesSettings.service_id(media_settings.INDEX_RQ))
        with open(shardmap_path) as shardmap_file:
            for line in shardmap_file:
                rbt = re.search(' (rbtorrent:[0-9a-h]+)', line)
                if rbt.group(1) is not None:
                    return rbt.group(1)
        return errors.TaskFailure("Failed to read imgrq shardmap file")

    def __get_rq_shard_name(self):
        shardmap_path = self.__get_nanny_shard_map(media_settings.ImagesSettings.service_id(media_settings.INDEX_RQ))
        shard_date = re.search('shardmap-[0-9]+-([0-9]+-[0-9]+)', shardmap_path)
        return 'imgsrq2baseidx-000-{}'.format(shard_date.group(1))

    def __get_shard_name(self, index_type, shard_prefix):
        if index_type == media_settings.INDEX_RQ:
            return self.__get_rq_shard_name()

        shard_prefix = "{}-".format(shard_prefix)
        shardmap_path = self.__get_nanny_shard_map(media_settings.ImagesSettings.service_id(index_type))

        if shardmap_path:
            with open(shardmap_path) as shardmap_file:
                for line in shardmap_file:
                    line = line.strip().split()
                    if len(line) < 2 or not line[1].startswith(shard_prefix):
                        continue
                    shard_name = line[1]
                    shard_attrs = shard_name.find("(")
                    if shard_attrs > 0:
                        shard_name = shard_name[:shard_attrs]
                    return shard_name
                else:
                    raise errors.TaskFailure("Failed to read shardmap name from shardmap")

        # fallback to cajuper index state file
        cajuper_index_state_path = self._get_nanny_cajuper_index_state(media_settings.ImagesSettings.service_id(index_type))
        if cajuper_index_state_path:
            state = None
            with open(cajuper_index_state_path) as index_state_file:
                for line in index_state_file:
                    tokens = line.strip().split('=')
                    if len(tokens) != 2:
                        continue
                    _, env_var = tokens[0].split()
                    if env_var == 'YT_STATE':
                        state = tokens[1]
            if not state:
                raise errors.TaskFailure('Can\'t read state from state file sandbox resource')

            for idx_type, shard_count in media_settings.ImagesSettings.FAKE_SHARDMAP_CONFIGURATIONS:
                if idx_type == index_type:
                    database_prefix = media_settings.ImagesSettings.basesearch_database_prefix(index_type)
                    shard_name = '{}-{:03d}-{}'.format(database_prefix, 0, state)
                    return shard_name

    def __get_nanny_client(self):
        return nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('IMAGES-SANDBOX', 'nanny-oauth-token')
        )

    @decorators.retries(max_tries=3, delay=10)
    def __get_nanny_instances(self, service_id):
        nanny_client = self.__get_nanny_client()
        return [
            (instance['hostname'], instance['port'], instance['container_hostname'])
            for instance in nanny_client.get_service_current_instances(service_id)['result']
        ]

    @decorators.retries(max_tries=3, delay=10)
    def __get_nanny_resource(self, service_id):
        nanny_client = self.__get_nanny_client()

        # search for active snapshot id
        # TODO: simplify after SWAT-2375
        current_state = nanny_client.get_service_current_state(service_id)
        if current_state['content']['summary']['value'] not in ('ONLINE', 'UPDATING', 'PREPARING'):
            raise errors.TaskFailure('Service {} is offline'.format(service_id))
        for snapshot in current_state['content']['active_snapshots']:
            if snapshot['state'] in ('ACTIVATING', 'GENERATING', 'PREPARING'):
                active_snapshot_id = current_state['content']['rollback_snapshot']['snapshot_id']
                break
            elif snapshot['state'] in ('ACTIVE'):
                active_snapshot_id = snapshot['snapshot_id']
                break
        else:
            raise errors.TaskFailure('Failed to find active snapshot')

        # retrieve information about shardmap
        runtime_attrs = nanny_client.get_history_runtime_attrs(active_snapshot_id)
        return runtime_attrs['content']['resources']

    @decorators.memoize
    def __get_sandbox_resource(self, task_id, resource_type):
        task_resources = apihelpers.list_task_resources(task_id, resource_type=resource_type)
        if not task_resources:
            raise errors.TaskFailure('Failed to find resource {} in task {}'.format(resource_type, task_id))
        return task_resources[0]

    @decorators.memoize
    def __get_nanny_shard_map(self, service_id):
        shardmap_data = self._get_nanny_shard_map_data(service_id)
        if not shardmap_data:
            return None
        shardmap_resource_id = self.__get_sandbox_resource(shardmap_data['task_id'], shardmap_data['resource_type']).id
        return self.sync_resource(shardmap_resource_id)

    @decorators.memoize
    def __get_nanny_shard_resource(self, service_id):
        shard_data = self.__get_nanny_resource(service_id)['sandbox_bsc_shard']
        return self.__get_sandbox_resource(shard_data['task_id'], shard_data['resource_type'])

    @decorators.memoize
    def __get_nanny_task(self, service_id, resource_type):
        sandbox_resources = self.__get_nanny_resource(service_id)['sandbox_files']
        resource_type = str(resource_type)

        for resource in sandbox_resources:
            if resource['resource_type'] == resource_type:
                return resource['task_id']

        raise errors.TaskFailure('Failed to find sandbox resource of type {}'.format(resource_type))

    @decorators.memoize
    def __get_nanny_file(self, service_id, resource_type):
        task_id = self.__get_nanny_task(service_id, resource_type)
        return self.__get_sandbox_resource(task_id, resource_type).id
