import os
import logging

from sandbox import sdk2
from sandbox.common.share import skynet_get
from sandbox.projects import resource_types


SAMPLE_FILENAME = 'sample'
SAMPLE_BIG_FILENAME = 'sample_big'
SHOOTING_URL = 'http://imtub-test.search.yandex.net/i?id='
BIG_THUMBS_SUFFIX = '-l'
RIM_THUMBS_TAG = 'rim_thumbs'
RIM_THUMBS_DEFAULT_TTL=128


class ImagesRimThumbsPlan(sdk2.Resource):
    rim_thumbs_state = sdk2.resource.Attributes.String('Rim thumbs state')
    valid_for_taas_acceptance = sdk2.resource.Attributes.Bool('Use this plan resource in taas acceptance')
    thumbs_type = sdk2.resource.Attributes.String('Plan thumbs type: main, big or both')
    any_arch = True
    executable = False
    releasable = False
    auto_backup = True


class ImagesBuildRimThumbsShootingPlan(sdk2.Task):
    """
    Create resource with rim thumbs sample shooting plan
    """

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup('Desired resource type') as resource_type:
            resource_type.values['ImagesRimThumbsPlan'] = resource_type.Value(value='ImagesRimThumbsPlan', default=True)
            resource_type.values['ImagesThumbsPlan'] = resource_type.Value(value='ImagesThumbsPlan')
        sample_filename = sdk2.parameters.String('Name of "sample" file', default=SAMPLE_FILENAME)
        sample_big_filename = sdk2.parameters.String('Name of "big sample" file', default=SAMPLE_BIG_FILENAME)
        shooting_url = sdk2.parameters.String('Shooting url', default=SHOOTING_URL)
        torrent = sdk2.parameters.String('Sample resource rbtorrent')
        big_thumbs_suffix = sdk2.parameters.String('Big thumbs suffix', default=BIG_THUMBS_SUFFIX)
        state = sdk2.parameters.String('Rim thumbs state')
        valid_for_taas_acceptance = sdk2.parameters.Bool('Use plan resource in taas acceptance', default=False)
        split = sdk2.parameters.Bool('Split "thumbids" and "big thumbids" tables', default=True)

    def on_execute(self):
        logging.info('Downloading shoting plan: {}'.format(self.Parameters.torrent))
        skynet_get(self.Parameters.torrent, '.')

        out_filename = os.path.join(os.getcwd(), 'fire_plan')
        out_filename_big = os.path.join(os.getcwd(), 'fire_plan_big')

        with open(out_filename, 'w') as fp:
            with open(out_filename_big, 'w') as fp_big:
                logging.info('Reading thumbids from plan file')
                with open(self.Parameters.sample_filename) as f:
                    fp.writelines('{}{}\n'.format(self.Parameters.shooting_url, line.strip()) for line in f)

                fp_2 = fp_big if self.Parameters.split else fp

                logging.info('Reading thumbids from "big" plan file')
                with open(self.Parameters.sample_big_filename) as big_f:
                    fp_2.writelines('{}{}{}\n'.format(self.Parameters.shooting_url, line.strip(),
                                                      self.Parameters.big_thumbs_suffix) for line in big_f)

        if self.Parameters.resource_type == 'ImagesRimThumbsPlan':
            def create_rim_result(type, state, filename, ttl, valid):
                logging.info('Creating IMAGES_RIM_THUMBS_PLAN resource, thumbs type "{}"'.format(type))
                resource = ImagesRimThumbsPlan(
                    self,
                    'Thumbs plan for rim thumbs state {}, thumbs type "{}"'.format(state, type),
                    os.path.basename(filename),
                    ttl=ttl
                )
                resource.rim_thumbs_state = state
                resource.valid_for_taas_acceptance = valid
                resource.thumbs_type = type
                resource_data = sdk2.ResourceData(resource)
                resource_data.ready()

            if self.Parameters.split:
                create_rim_result("main", self.Parameters.state, out_filename, RIM_THUMBS_DEFAULT_TTL, self.Parameters.valid_for_taas_acceptance)
                create_rim_result("big", self.Parameters.state, out_filename_big, RIM_THUMBS_DEFAULT_TTL, self.Parameters.valid_for_taas_acceptance)
            else:
                create_rim_result("both", self.Parameters.state, out_filename, RIM_THUMBS_DEFAULT_TTL, self.Parameters.valid_for_taas_acceptance)
        elif self.Parameters.resource_type == 'ImagesThumbsPlan':
            def create_legacy_result(type, state, filename, ttl):
                logging.info('Creating IMAGES_THUMBS_PLAN resource, thumbs type "{}"'.format(type))
                resource = resource_types.IMAGES_THUMBS_PLAN(
                    self,
                    'Thumbs plan for rim thumbs state {}, thumbs type "{}"'.format(state, type),
                    os.path.basename(filename),
                    ttl=ttl
                )
                resource_data = sdk2.ResourceData(resource)
                resource_data.ready()

            if self.Parameters.split:
                create_legacy_result("main", self.Parameters.state, out_filename, RIM_THUMBS_DEFAULT_TTL)
                create_legacy_result("big", self.Parameters.state, out_filename_big, RIM_THUMBS_DEFAULT_TTL)
            else:
                create_legacy_result("both", self.Parameters.state, out_filename, RIM_THUMBS_DEFAULT_TTL)
