import logging
import sandbox.common.types.task as ctt
import sandbox.projects.images.resource_types as images_resource_types

from sandbox import sdk2

from sandbox.sdk2.paths import copy_path

from sandbox.projects import resource_types
from sandbox.projects.images.robot.ImagesRobotBase import ImagesRobotBase
from sandbox.projects.images.robot.ImagesComposeRobotBundle import ImagesComposeRobotBundle
from sandbox.projects.images.robot.resources import ImagesRobotResourcesList


IMAGES_SEARCH_BASE_DEPLOY_BUNDLE_NAME = "IMAGES_SEARCH_BASE_DEPLOY_BUNDLE"
IMAGES_SEARCH_BASE_DEPLOY_BUNDLE_EXPERIMENTAL_NAME = "IMAGES_SEARCH_BASE_DEPLOY_BUNDLE_EXPERIMENTAL"


def merge_dictionaries(x, y):
    result = x.copy()
    result.update(y)
    return result


class ImagesBuildMainRobot(ImagesRobotBase, sdk2.Task):
    COMMON_SEARCH_BASE_DEPLOY_BUNDLE_MAPPING = {
        "bin/indexmerge": "indexmerge",
        "bin/main-index-install.sh": "install.sh",
        "bin/main_index_download": "shard_download",
        "bin/inverted_index_download": "inverted_index_download",
        "bin/shard_checker": "shard_checker",
        "bin/shardwriter": "shardwriter",
        "bin/wrangler": "wrangler",
    }

    BUNDLES = {
        IMAGES_SEARCH_BASE_DEPLOY_BUNDLE_NAME: (
            "imgsbase",
            merge_dictionaries(COMMON_SEARCH_BASE_DEPLOY_BUNDLE_MAPPING,
                               {"bin/search_embedding_download": "search_embedding_download"}),
        ),
        IMAGES_SEARCH_BASE_DEPLOY_BUNDLE_EXPERIMENTAL_NAME: (
            "imgsbase",
            merge_dictionaries(COMMON_SEARCH_BASE_DEPLOY_BUNDLE_MAPPING,
                               {"bin/search_embedding_experimental_download": "search_embedding_download"}),
        ),
        "IMAGES_META_DEPLOY_BUNDLE": (
            "imgmmeta",
            {
                "bin/main_mmeta_download": "shard_download",
                "bin/shardwriter": "shardwriter",
            },
        ),
        "IMAGES_MAIN_IMTUB_DEPLOY_BUNDLE": (
            "imgsth",
            {
                "bin/main_imtub_download": "shard_download",
                "bin/shardwriter": "shardwriter",
                "bin/thdb": "thdb",
                "bin/main-imtub-install.sh": "install.sh",
            },
        ),
    }

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl("Arcadia Url", required=True)
        build_package = sdk2.parameters.Task("Package build task",
                                             required=False,
                                             task_type="IMAGES_BUILD_CM_PACKAGE")
        build_quick_package = sdk2.parameters.Task("Quick package build task",
                                                   required=False,
                                                   task_type="IMAGES_BUILD_QUICK_PACKAGE")

        release_quick_package = sdk2.parameters.Bool("Release Quick package", required=False, default=False)

        build_configs = sdk2.parameters.Task("Configs build task",
                                             required=False,
                                             task_type="GET_IMAGES_MR_INDEX_CONFIG")
        experimental_build = sdk2.parameters.Bool("Build experimental bundle", required=False, default=False)

    class Requirements(sdk2.Requirements):
        disk_space = 50 * 1024

    def on_execute(self):
        with self.memoize_stage.package:
            self.__package()
            self.__quick_package()

        with self.memoize_stage.configs:
            self.__configs()

        with self.memoize_stage.wait_build_tasks:
            raise sdk2.WaitTask([self.Context.build_task_id,
                                 self.Context.build_quick_task_id,
                                 self.Context.config_task_id],
                                ctt.Status.Group.FINISH,
                                wait_all=True)

        package_resource = sdk2.Resource.find(
            images_resource_types.IMAGES_CM_PACKAGE,
            task_id=self.Context.build_task_id).first()

        sw_config_resource = sdk2.Resource.find(
            resource_types.IMAGES_SHARDWRITER_CONFIG,
            task_id=self.Context.config_task_id).first()

        mrindex_config_resource = sdk2.Resource.find(
            resource_types.IMAGES_MR_INDEX_CONFIG,
            task_id=self.Context.config_task_id).first()

        with self.memoize_stage.copy_resouces:
            self.__copy_resouces(package_resource,
                                 mrindex_config_resource,
                                 sw_config_resource)

        with self.memoize_stage.bundles:
            logging.info("Start creating bundles")
            self.__create_bundles(package_resource, sw_config_resource)

        resources_list = [package_resource, mrindex_config_resource,
                          sw_config_resource]

        self.__add_bundles_to_list(resources_list)
        self.__create_robot_resouces_list(resources_list)

    def on_release(self, additional_parameters):
        self.__release_task(self.Context.build_task_id,
                            additional_parameters['release_status'])
        self.__release_task(self.Context.build_quick_task_id,
                            additional_parameters['release_status'], release=self.Parameters.release_quick_package)
        self.__release_task(self.Context.config_task_id,
                            additional_parameters['release_status'])

        sdk2.Task.on_release(self, additional_parameters)

    def __get_arcadia_url(self):
        if self.Context.checkout_arcadia_from_url:
            return self.Context.checkout_arcadia_from_url
        else:
            return self.Parameters.arcadia_url

    def __package(self):
        task_id = None
        if not self.Parameters.build_package:
            logging.info("Start building package")
            task_id = self.__run_task_sdk1("IMAGES_BUILD_CM_PACKAGE", description="Build CM package",
                                           checkout_arcadia_from_url=self.__get_arcadia_url(),
                                           kill_timeout=6 * 60 * 60)
        else:
            task_id = self.Parameters.build_package.id

        self.Context.build_task_id = task_id
        self.Context.save()

    def __quick_package(self):
        task_id = None
        if not self.Parameters.build_quick_package:
            logging.info("Start building quick package")
            task_id = self.__run_task_sdk1("IMAGES_BUILD_QUICK_PACKAGE", description="Build Quick package",
                                           checkout_arcadia_from_url=self.__get_arcadia_url(),
                                           kill_timeout=6 * 60 * 60)
        else:
            task_id = self.Parameters.build_quick_package.id

        self.Context.build_quick_task_id = task_id
        self.Context.save()

    def __configs(self):
        task_id = None
        if not self.Parameters.build_configs:
            logging.info("Start building configs")
            task_id = self.__run_task_sdk1("GET_IMAGES_MR_INDEX_CONFIG", description="Build config")
        else:
            task_id = self.Parameters.build_configs.id

        self.Context.config_task_id = task_id
        self.Context.save()

    def __copy_resouces(self, package_resource, mrindex_config_resource, sw_config_resource):
        package = sdk2.ResourceData(package_resource)
        mrindex_config = sdk2.ResourceData(mrindex_config_resource)
        sw_config = sdk2.ResourceData(sw_config_resource)

        package_copied = sdk2.ResourceData(images_resource_types.IMAGES_CM_PACKAGE(
            self, "IMAGES_CM_PACKAGE copied from child task", 'package_data'
        ))
        mrindex_config_copied = sdk2.ResourceData(resource_types.IMAGES_MR_INDEX_CONFIG(
            self, "IMAGES_MR_INDEX_CONFIG copied from child task", 'mrindex_config_data'
        ))
        sw_config_copied = sdk2.ResourceData(resource_types.IMAGES_SHARDWRITER_CONFIG(
            self, "IMAGES_SW_INDEX_CONFIG copied from child task", 'sw_config_data'
        ))

        copy_path(str(package.path), str(package_copied.path))
        copy_path(str(mrindex_config.path), str(mrindex_config_copied.path))
        copy_path(str(sw_config.path), str(sw_config_copied.path))

        package_copied.ready()
        mrindex_config_copied.ready()
        sw_config_copied.ready()

    def __create_bundles(self, package_resource, config_resource):
        self.Context.bundle_task_ids = []
        self.Context.bundle_resouce_ids = []
        for name, (path, mapping) in ImagesBuildMainRobot.BUNDLES.iteritems():
            original_name = name
            if name == IMAGES_SEARCH_BASE_DEPLOY_BUNDLE_EXPERIMENTAL_NAME and self.Parameters.experimental_build:
                name = IMAGES_SEARCH_BASE_DEPLOY_BUNDLE_NAME
            elif name == IMAGES_SEARCH_BASE_DEPLOY_BUNDLE_NAME and not self.Parameters.experimental_build:
                pass
            elif IMAGES_SEARCH_BASE_DEPLOY_BUNDLE_NAME in name:
                continue

            logging.info("Create task for {}.".format(original_name))
            resource_type = getattr(resource_types, name)
            bundle = resource_type(self, "Deploy bundle", path)
            self.Context.bundle_resouce_ids.append(bundle.id)
            bundle_task = ImagesComposeRobotBundle(
                self,
                description="Build bundle {}".format(original_name),
                package=package_resource,
                sw_config=config_resource,
                package_to_bundle_files=mapping,
                bundle=bundle
            )
            bundle_task.enqueue()
            self.Context.bundle_task_ids.append(bundle_task.id)
            self.Context.save()
        raise sdk2.WaitTask(self.Context.bundle_task_ids,
                            ctt.Status.Group.FINISH,
                            wait_all=True)

    def __create_robot_resouces_list(self, resources_list):
        resourse = ImagesRobotResourcesList(self,
                                            "List of resouses id",
                                            "resource_list")
        resource_data = sdk2.ResourceData(resourse)

        resource_data.path.write_bytes(self.__resource_list(resources_list))
        resource_data.ready()

    def __add_bundles_to_list(self, resources_list):
        for id in self.Context.bundle_resouce_ids:
            resource = sdk2.Resource.find(id=id).first()
            resources_list.append(resource)

    def __resource_list(self, resouces):
        return "\n".join(["{}={}".format(resource.type.name, resource.id)
                          for resource in resouces])

    def __run_task_sdk1(self, class_name, **kwargs):
        task_class = sdk2.Task[class_name]
        task = task_class(self, **kwargs)
        task.enqueue()
        return task.id

    def __release_task(self, task_id, release_status, release=True):
        if release:
            self.server.release(
                task_id=task_id,
                type=release_status,
                subject='Images robot main package',
                comments="Relese to {}".format(release_status),
            )
